// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package drive provides access to the Google Drive API.
//
// For product documentation, see: https://developers.google.com/workspace/drive/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/drive/v3"
//	...
//	ctx := context.Background()
//	driveService, err := drive.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	driveService, err := drive.NewService(ctx, option.WithScopes(drive.DriveScriptsScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	driveService, err := drive.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	driveService, err := drive.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package drive // import "google.golang.org/api/drive/v3"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "drive:v3"
const apiName = "drive"
const apiVersion = "v3"
const basePath = "https://www.googleapis.com/drive/v3/"
const basePathTemplate = "https://www.UNIVERSE_DOMAIN/drive/v3/"
const mtlsBasePath = "https://www.mtls.googleapis.com/drive/v3/"

// OAuth2 scopes used by this API.
const (
	// See, edit, create, and delete all of your Google Drive files
	DriveScope = "https://www.googleapis.com/auth/drive"

	// See, create, and delete its own configuration data in your Google Drive
	DriveAppdataScope = "https://www.googleapis.com/auth/drive.appdata"

	// View your Google Drive apps
	DriveAppsReadonlyScope = "https://www.googleapis.com/auth/drive.apps.readonly"

	// See, edit, create, and delete only the specific Google Drive files you use
	// with this app
	DriveFileScope = "https://www.googleapis.com/auth/drive.file"

	// See and download your Google Drive files that were created or edited by
	// Google Meet.
	DriveMeetReadonlyScope = "https://www.googleapis.com/auth/drive.meet.readonly"

	// View and manage metadata of files in your Google Drive
	DriveMetadataScope = "https://www.googleapis.com/auth/drive.metadata"

	// See information about your Google Drive files
	DriveMetadataReadonlyScope = "https://www.googleapis.com/auth/drive.metadata.readonly"

	// View the photos, videos and albums in your Google Photos
	DrivePhotosReadonlyScope = "https://www.googleapis.com/auth/drive.photos.readonly"

	// See and download all your Google Drive files
	DriveReadonlyScope = "https://www.googleapis.com/auth/drive.readonly"

	// Modify your Google Apps Script scripts' behavior
	DriveScriptsScope = "https://www.googleapis.com/auth/drive.scripts"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/drive",
		"https://www.googleapis.com/auth/drive.appdata",
		"https://www.googleapis.com/auth/drive.apps.readonly",
		"https://www.googleapis.com/auth/drive.file",
		"https://www.googleapis.com/auth/drive.meet.readonly",
		"https://www.googleapis.com/auth/drive.metadata",
		"https://www.googleapis.com/auth/drive.metadata.readonly",
		"https://www.googleapis.com/auth/drive.photos.readonly",
		"https://www.googleapis.com/auth/drive.readonly",
		"https://www.googleapis.com/auth/drive.scripts",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.About = NewAboutService(s)
	s.Accessproposals = NewAccessproposalsService(s)
	s.Apps = NewAppsService(s)
	s.Changes = NewChangesService(s)
	s.Channels = NewChannelsService(s)
	s.Comments = NewCommentsService(s)
	s.Drives = NewDrivesService(s)
	s.Files = NewFilesService(s)
	s.Operations = NewOperationsService(s)
	s.Permissions = NewPermissionsService(s)
	s.Replies = NewRepliesService(s)
	s.Revisions = NewRevisionsService(s)
	s.Teamdrives = NewTeamdrivesService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	About *AboutService

	Accessproposals *AccessproposalsService

	Apps *AppsService

	Changes *ChangesService

	Channels *ChannelsService

	Comments *CommentsService

	Drives *DrivesService

	Files *FilesService

	Operations *OperationsService

	Permissions *PermissionsService

	Replies *RepliesService

	Revisions *RevisionsService

	Teamdrives *TeamdrivesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAboutService(s *Service) *AboutService {
	rs := &AboutService{s: s}
	return rs
}

type AboutService struct {
	s *Service
}

func NewAccessproposalsService(s *Service) *AccessproposalsService {
	rs := &AccessproposalsService{s: s}
	return rs
}

type AccessproposalsService struct {
	s *Service
}

func NewAppsService(s *Service) *AppsService {
	rs := &AppsService{s: s}
	return rs
}

type AppsService struct {
	s *Service
}

func NewChangesService(s *Service) *ChangesService {
	rs := &ChangesService{s: s}
	return rs
}

type ChangesService struct {
	s *Service
}

func NewChannelsService(s *Service) *ChannelsService {
	rs := &ChannelsService{s: s}
	return rs
}

type ChannelsService struct {
	s *Service
}

func NewCommentsService(s *Service) *CommentsService {
	rs := &CommentsService{s: s}
	return rs
}

type CommentsService struct {
	s *Service
}

func NewDrivesService(s *Service) *DrivesService {
	rs := &DrivesService{s: s}
	return rs
}

type DrivesService struct {
	s *Service
}

func NewFilesService(s *Service) *FilesService {
	rs := &FilesService{s: s}
	return rs
}

type FilesService struct {
	s *Service
}

func NewOperationsService(s *Service) *OperationsService {
	rs := &OperationsService{s: s}
	return rs
}

type OperationsService struct {
	s *Service
}

func NewPermissionsService(s *Service) *PermissionsService {
	rs := &PermissionsService{s: s}
	return rs
}

type PermissionsService struct {
	s *Service
}

func NewRepliesService(s *Service) *RepliesService {
	rs := &RepliesService{s: s}
	return rs
}

type RepliesService struct {
	s *Service
}

func NewRevisionsService(s *Service) *RevisionsService {
	rs := &RevisionsService{s: s}
	return rs
}

type RevisionsService struct {
	s *Service
}

func NewTeamdrivesService(s *Service) *TeamdrivesService {
	rs := &TeamdrivesService{s: s}
	return rs
}

type TeamdrivesService struct {
	s *Service
}

// About: Information about the user, the user's Drive, and system
// capabilities.
type About struct {
	// AppInstalled: Whether the user has installed the requesting app.
	AppInstalled bool `json:"appInstalled,omitempty"`
	// CanCreateDrives: Whether the user can create shared drives.
	CanCreateDrives bool `json:"canCreateDrives,omitempty"`
	// CanCreateTeamDrives: Deprecated: Use `canCreateDrives` instead.
	CanCreateTeamDrives bool `json:"canCreateTeamDrives,omitempty"`
	// DriveThemes: A list of themes that are supported for shared drives.
	DriveThemes []*AboutDriveThemes `json:"driveThemes,omitempty"`
	// ExportFormats: A map of source MIME type to possible targets for all
	// supported exports.
	ExportFormats map[string][]string `json:"exportFormats,omitempty"`
	// FolderColorPalette: The currently supported folder colors as RGB hex
	// strings.
	FolderColorPalette []string `json:"folderColorPalette,omitempty"`
	// ImportFormats: A map of source MIME type to possible targets for all
	// supported imports.
	ImportFormats map[string][]string `json:"importFormats,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#about".
	Kind string `json:"kind,omitempty"`
	// MaxImportSizes: A map of maximum import sizes by MIME type, in bytes.
	MaxImportSizes map[string]string `json:"maxImportSizes,omitempty"`
	// MaxUploadSize: The maximum upload size in bytes.
	MaxUploadSize int64 `json:"maxUploadSize,omitempty,string"`
	// StorageQuota: The user's storage quota limits and usage. For users that are
	// part of an organization with pooled storage, information about the limit and
	// usage across all services is for the organization, rather than the
	// individual user. All fields are measured in bytes.
	StorageQuota *AboutStorageQuota `json:"storageQuota,omitempty"`
	// TeamDriveThemes: Deprecated: Use `driveThemes` instead.
	TeamDriveThemes []*AboutTeamDriveThemes `json:"teamDriveThemes,omitempty"`
	// User: The authenticated user.
	User *User `json:"user,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AppInstalled") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AppInstalled") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s About) MarshalJSON() ([]byte, error) {
	type NoMethod About
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutDriveThemes struct {
	// BackgroundImageLink: A link to this theme's background image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// ColorRgb: The color of this theme as an RGB hex string.
	ColorRgb string `json:"colorRgb,omitempty"`
	// Id: The ID of the theme.
	Id string `json:"id,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageLink") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutDriveThemes) MarshalJSON() ([]byte, error) {
	type NoMethod AboutDriveThemes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AboutStorageQuota: The user's storage quota limits and usage. For users that
// are part of an organization with pooled storage, information about the limit
// and usage across all services is for the organization, rather than the
// individual user. All fields are measured in bytes.
type AboutStorageQuota struct {
	// Limit: The usage limit, if applicable. This will not be present if the user
	// has unlimited storage. For users that are part of an organization with
	// pooled storage, this is the limit for the organization, rather than the
	// individual user.
	Limit int64 `json:"limit,omitempty,string"`
	// Usage: The total usage across all services. For users that are part of an
	// organization with pooled storage, this is the usage across all services for
	// the organization, rather than the individual user.
	Usage int64 `json:"usage,omitempty,string"`
	// UsageInDrive: The usage by all files in Google Drive.
	UsageInDrive int64 `json:"usageInDrive,omitempty,string"`
	// UsageInDriveTrash: The usage by trashed files in Google Drive.
	UsageInDriveTrash int64 `json:"usageInDriveTrash,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Limit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Limit") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutStorageQuota) MarshalJSON() ([]byte, error) {
	type NoMethod AboutStorageQuota
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutTeamDriveThemes struct {
	// BackgroundImageLink: Deprecated: Use `driveThemes/backgroundImageLink`
	// instead.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// ColorRgb: Deprecated: Use `driveThemes/colorRgb` instead.
	ColorRgb string `json:"colorRgb,omitempty"`
	// Id: Deprecated: Use `driveThemes/id` instead.
	Id string `json:"id,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageLink") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutTeamDriveThemes) MarshalJSON() ([]byte, error) {
	type NoMethod AboutTeamDriveThemes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessProposal: Manage outstanding access proposals on a file.
type AccessProposal struct {
	// CreateTime: The creation time.
	CreateTime string `json:"createTime,omitempty"`
	// FileId: The file ID that the proposal for access is on.
	FileId string `json:"fileId,omitempty"`
	// ProposalId: The ID of the access proposal.
	ProposalId string `json:"proposalId,omitempty"`
	// RecipientEmailAddress: The email address of the user that will receive
	// permissions, if accepted.
	RecipientEmailAddress string `json:"recipientEmailAddress,omitempty"`
	// RequestMessage: The message that the requester added to the proposal.
	RequestMessage string `json:"requestMessage,omitempty"`
	// RequesterEmailAddress: The email address of the requesting user.
	RequesterEmailAddress string `json:"requesterEmailAddress,omitempty"`
	// RolesAndViews: A wrapper for the role and view of an access proposal. For
	// more information, see Roles and permissions
	// (https://developers.google.com/workspace/drive/api/guides/ref-roles).
	RolesAndViews []*AccessProposalRoleAndView `json:"rolesAndViews,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessProposal) MarshalJSON() ([]byte, error) {
	type NoMethod AccessProposal
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessProposalRoleAndView: A wrapper for the role and view of an access
// proposal. For more information, see Roles and permissions
// (https://developers.google.com/workspace/drive/api/guides/ref-roles).
type AccessProposalRoleAndView struct {
	// Role: The role that was proposed by the requester. The supported values are:
	// * `writer` * `commenter` * `reader`
	Role string `json:"role,omitempty"`
	// View: Indicates the view for this access proposal. Only populated for
	// proposals that belong to a view. Only `published` is supported.
	View string `json:"view,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Role") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Role") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessProposalRoleAndView) MarshalJSON() ([]byte, error) {
	type NoMethod AccessProposalRoleAndView
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// App: The `apps` resource provides a list of apps that a user has installed,
// with information about each app's supported MIME types, file extensions, and
// other details. Some resource methods (such as `apps.get`) require an
// `appId`. Use the `apps.list` method to retrieve the ID for an installed
// application.
type App struct {
	// Authorized: Whether the app is authorized to access data on the user's
	// Drive.
	Authorized bool `json:"authorized,omitempty"`
	// CreateInFolderTemplate: The template URL to create a file with this app in a
	// given folder. The template contains the {folderId} to be replaced by the
	// folder ID house the new file.
	CreateInFolderTemplate string `json:"createInFolderTemplate,omitempty"`
	// CreateUrl: The URL to create a file with this app.
	CreateUrl string `json:"createUrl,omitempty"`
	// HasDriveWideScope: Whether the app has Drive-wide scope. An app with
	// Drive-wide scope can access all files in the user's Drive.
	HasDriveWideScope bool `json:"hasDriveWideScope,omitempty"`
	// Icons: The various icons for the app.
	Icons []*AppIcons `json:"icons,omitempty"`
	// Id: The ID of the app.
	Id string `json:"id,omitempty"`
	// Installed: Whether the app is installed.
	Installed bool `json:"installed,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#app".
	Kind string `json:"kind,omitempty"`
	// LongDescription: A long description of the app.
	LongDescription string `json:"longDescription,omitempty"`
	// Name: The name of the app.
	Name string `json:"name,omitempty"`
	// ObjectType: The type of object this app creates such as a Chart. If empty,
	// the app name should be used instead.
	ObjectType string `json:"objectType,omitempty"`
	// OpenUrlTemplate: The template URL for opening files with this app. The
	// template contains {ids} or {exportIds} to be replaced by the actual file
	// IDs. For more information, see Open Files for the full documentation.
	OpenUrlTemplate string `json:"openUrlTemplate,omitempty"`
	// PrimaryFileExtensions: The list of primary file extensions.
	PrimaryFileExtensions []string `json:"primaryFileExtensions,omitempty"`
	// PrimaryMimeTypes: The list of primary MIME types.
	PrimaryMimeTypes []string `json:"primaryMimeTypes,omitempty"`
	// ProductId: The ID of the product listing for this app.
	ProductId string `json:"productId,omitempty"`
	// ProductUrl: A link to the product listing for this app.
	ProductUrl string `json:"productUrl,omitempty"`
	// SecondaryFileExtensions: The list of secondary file extensions.
	SecondaryFileExtensions []string `json:"secondaryFileExtensions,omitempty"`
	// SecondaryMimeTypes: The list of secondary MIME types.
	SecondaryMimeTypes []string `json:"secondaryMimeTypes,omitempty"`
	// ShortDescription: A short description of the app.
	ShortDescription string `json:"shortDescription,omitempty"`
	// SupportsCreate: Whether this app supports creating objects.
	SupportsCreate bool `json:"supportsCreate,omitempty"`
	// SupportsImport: Whether this app supports importing from Google Docs.
	SupportsImport bool `json:"supportsImport,omitempty"`
	// SupportsMultiOpen: Whether this app supports opening more than one file.
	SupportsMultiOpen bool `json:"supportsMultiOpen,omitempty"`
	// SupportsOfflineCreate: Whether this app supports creating files when
	// offline.
	SupportsOfflineCreate bool `json:"supportsOfflineCreate,omitempty"`
	// UseByDefault: Whether the app is selected as the default handler for the
	// types it supports.
	UseByDefault bool `json:"useByDefault,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Authorized") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Authorized") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s App) MarshalJSON() ([]byte, error) {
	type NoMethod App
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AppIcons struct {
	// Category: Category of the icon. Allowed values are: * `application` - The
	// icon for the application. * `document` - The icon for a file associated with
	// the app. * `documentShared` - The icon for a shared file associated with the
	// app.
	Category string `json:"category,omitempty"`
	// IconUrl: URL for the icon.
	IconUrl string `json:"iconUrl,omitempty"`
	// Size: Size of the icon. Represented as the maximum of the width and height.
	Size int64 `json:"size,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Category") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Category") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AppIcons) MarshalJSON() ([]byte, error) {
	type NoMethod AppIcons
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AppList: A list of third-party applications which the user has installed or
// given access to Google Drive.
type AppList struct {
	// DefaultAppIds: The list of app IDs that the user has specified to use by
	// default. The list is in reverse-priority order (lowest to highest).
	DefaultAppIds []string `json:"defaultAppIds,omitempty"`
	// Items: The list of apps.
	Items []*App `json:"items,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#appList".
	Kind string `json:"kind,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DefaultAppIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DefaultAppIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AppList) MarshalJSON() ([]byte, error) {
	type NoMethod AppList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Change: A change to a file or shared drive.
type Change struct {
	// ChangeType: The type of the change. Possible values are `file` and `drive`.
	ChangeType string `json:"changeType,omitempty"`
	// Drive: The updated state of the shared drive. Present if the changeType is
	// drive, the user is still a member of the shared drive, and the shared drive
	// has not been deleted.
	Drive *Drive `json:"drive,omitempty"`
	// DriveId: The ID of the shared drive associated with this change.
	DriveId string `json:"driveId,omitempty"`
	// File: The updated state of the file. Present if the type is file and the
	// file has not been removed from this list of changes.
	File *File `json:"file,omitempty"`
	// FileId: The ID of the file which has changed.
	FileId string `json:"fileId,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#change".
	Kind string `json:"kind,omitempty"`
	// Removed: Whether the file or shared drive has been removed from this list of
	// changes, for example by deletion or loss of access.
	Removed bool `json:"removed,omitempty"`
	// TeamDrive: Deprecated: Use `drive` instead.
	TeamDrive *TeamDrive `json:"teamDrive,omitempty"`
	// TeamDriveId: Deprecated: Use `driveId` instead.
	TeamDriveId string `json:"teamDriveId,omitempty"`
	// Time: The time of this change (RFC 3339 date-time).
	Time string `json:"time,omitempty"`
	// Type: Deprecated: Use `changeType` instead.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChangeType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChangeType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Change) MarshalJSON() ([]byte, error) {
	type NoMethod Change
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChangeList: A list of changes for a user.
type ChangeList struct {
	// Changes: The list of changes. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Changes []*Change `json:"changes,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#changeList".
	Kind string `json:"kind,omitempty"`
	// NewStartPageToken: The starting page token for future changes. This will be
	// present only if the end of the current changes list has been reached. The
	// page token doesn't expire.
	NewStartPageToken string `json:"newStartPageToken,omitempty"`
	// NextPageToken: The page token for the next page of changes. This will be
	// absent if the end of the changes list has been reached. The page token
	// doesn't expire.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Changes") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Changes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChangeList) MarshalJSON() ([]byte, error) {
	type NoMethod ChangeList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Channel: A notification channel used to watch for resource changes.
type Channel struct {
	// Address: The address where notifications are delivered for this channel.
	Address string `json:"address,omitempty"`
	// Expiration: Date and time of notification channel expiration, expressed as a
	// Unix timestamp, in milliseconds. Optional.
	Expiration int64 `json:"expiration,omitempty,string"`
	// Id: A UUID or similar unique string that identifies this channel.
	Id string `json:"id,omitempty"`
	// Kind: Identifies this as a notification channel used to watch for changes to
	// a resource, which is `api#channel`.
	Kind string `json:"kind,omitempty"`
	// Params: Additional parameters controlling delivery channel behavior.
	// Optional.
	Params map[string]string `json:"params,omitempty"`
	// Payload: A Boolean value to indicate whether payload is wanted. Optional.
	Payload bool `json:"payload,omitempty"`
	// ResourceId: An opaque ID that identifies the resource being watched on this
	// channel. Stable across different API versions.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceUri: A version-specific identifier for the watched resource.
	ResourceUri string `json:"resourceUri,omitempty"`
	// Token: An arbitrary string delivered to the target address with each
	// notification delivered over this channel. Optional.
	Token string `json:"token,omitempty"`
	// Type: The type of delivery mechanism used for this channel. Valid values are
	// "web_hook" or "webhook".
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Channel) MarshalJSON() ([]byte, error) {
	type NoMethod Channel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Comment: A comment on a file. Some resource methods (such as
// `comments.update`) require a `commentId`. Use the `comments.list` method to
// retrieve the ID for a comment in a file.
type Comment struct {
	// Anchor: A region of the document represented as a JSON string. For details
	// on defining anchor properties, refer to Manage comments and replies
	// (https://developers.google.com/workspace/drive/api/v3/manage-comments).
	Anchor string `json:"anchor,omitempty"`
	// Author: Output only. The author of the comment. The author's email address
	// and permission ID will not be populated.
	Author *User `json:"author,omitempty"`
	// Content: The plain text content of the comment. This field is used for
	// setting the content, while `htmlContent` should be displayed.
	Content string `json:"content,omitempty"`
	// CreatedTime: The time at which the comment was created (RFC 3339 date-time).
	CreatedTime string `json:"createdTime,omitempty"`
	// Deleted: Output only. Whether the comment has been deleted. A deleted
	// comment has no content.
	Deleted bool `json:"deleted,omitempty"`
	// HtmlContent: Output only. The content of the comment with HTML formatting.
	HtmlContent string `json:"htmlContent,omitempty"`
	// Id: Output only. The ID of the comment.
	Id string `json:"id,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#comment".
	Kind string `json:"kind,omitempty"`
	// ModifiedTime: The last time the comment or any of its replies was modified
	// (RFC 3339 date-time).
	ModifiedTime string `json:"modifiedTime,omitempty"`
	// QuotedFileContent: The file content to which the comment refers, typically
	// within the anchor region. For a text file, for example, this would be the
	// text at the location of the comment.
	QuotedFileContent *CommentQuotedFileContent `json:"quotedFileContent,omitempty"`
	// Replies: Output only. The full list of replies to the comment in
	// chronological order.
	Replies []*Reply `json:"replies,omitempty"`
	// Resolved: Output only. Whether the comment has been resolved by one of its
	// replies.
	Resolved bool `json:"resolved,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Anchor") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Anchor") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Comment) MarshalJSON() ([]byte, error) {
	type NoMethod Comment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentQuotedFileContent: The file content to which the comment refers,
// typically within the anchor region. For a text file, for example, this would
// be the text at the location of the comment.
type CommentQuotedFileContent struct {
	// MimeType: The MIME type of the quoted content.
	MimeType string `json:"mimeType,omitempty"`
	// Value: The quoted content itself. This is interpreted as plain text if set
	// through the API.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MimeType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MimeType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentQuotedFileContent) MarshalJSON() ([]byte, error) {
	type NoMethod CommentQuotedFileContent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentList: A list of comments on a file.
type CommentList struct {
	// Comments: The list of comments. If nextPageToken is populated, then this
	// list may be incomplete and an additional page of results should be fetched.
	Comments []*Comment `json:"comments,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#commentList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of comments. This will be
	// absent if the end of the comments list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results. The page token is typically valid
	// for several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Comments") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Comments") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentList) MarshalJSON() ([]byte, error) {
	type NoMethod CommentList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ContentRestriction: A restriction for accessing the content of the file.
type ContentRestriction struct {
	// OwnerRestricted: Whether the content restriction can only be modified or
	// removed by a user who owns the file. For files in shared drives, any user
	// with `organizer` capabilities can modify or remove this content restriction.
	OwnerRestricted bool `json:"ownerRestricted,omitempty"`
	// ReadOnly: Whether the content of the file is read-only. If a file is
	// read-only, a new revision of the file may not be added, comments may not be
	// added or modified, and the title of the file may not be modified.
	ReadOnly bool `json:"readOnly,omitempty"`
	// Reason: Reason for why the content of the file is restricted. This is only
	// mutable on requests that also set `readOnly=true`.
	Reason string `json:"reason,omitempty"`
	// RestrictingUser: Output only. The user who set the content restriction. Only
	// populated if `readOnly=true`.
	RestrictingUser *User `json:"restrictingUser,omitempty"`
	// RestrictionTime: The time at which the content restriction was set
	// (formatted RFC 3339 timestamp). Only populated if readOnly is true.
	RestrictionTime string `json:"restrictionTime,omitempty"`
	// SystemRestricted: Output only. Whether the content restriction was applied
	// by the system, for example due to an esignature. Users cannot modify or
	// remove system restricted content restrictions.
	SystemRestricted bool `json:"systemRestricted,omitempty"`
	// Type: Output only. The type of the content restriction. Currently the only
	// possible value is `globalContentRestriction`.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OwnerRestricted") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OwnerRestricted") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ContentRestriction) MarshalJSON() ([]byte, error) {
	type NoMethod ContentRestriction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DownloadRestriction: A restriction for copy and download of the file.
type DownloadRestriction struct {
	// RestrictedForReaders: Whether download and copy is restricted for readers.
	RestrictedForReaders bool `json:"restrictedForReaders,omitempty"`
	// RestrictedForWriters: Whether download and copy is restricted for writers.
	// If `true`, download is also restricted for readers.
	RestrictedForWriters bool `json:"restrictedForWriters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RestrictedForReaders") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RestrictedForReaders") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DownloadRestriction) MarshalJSON() ([]byte, error) {
	type NoMethod DownloadRestriction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DownloadRestrictionsMetadata: Download restrictions applied to the file.
type DownloadRestrictionsMetadata struct {
	// EffectiveDownloadRestrictionWithContext: Output only. The effective download
	// restriction applied to this file. This considers all restriction settings
	// and DLP rules.
	EffectiveDownloadRestrictionWithContext *DownloadRestriction `json:"effectiveDownloadRestrictionWithContext,omitempty"`
	// ItemDownloadRestriction: The download restriction of the file applied
	// directly by the owner or organizer. This doesn't take into account shared
	// drive settings or DLP rules.
	ItemDownloadRestriction *DownloadRestriction `json:"itemDownloadRestriction,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "EffectiveDownloadRestrictionWithContext") to unconditionally include in API
	// requests. By default, fields with empty or default values are omitted from
	// API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g.
	// "EffectiveDownloadRestrictionWithContext") to include in API requests with
	// the JSON null value. By default, fields with empty values are omitted from
	// API requests. See https://pkg.go.dev/google.golang.org/api#hdr-NullFields
	// for more details.
	NullFields []string `json:"-"`
}

func (s DownloadRestrictionsMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod DownloadRestrictionsMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Drive: Representation of a shared drive. Some resource methods (such as
// `drives.update`) require a `driveId`. Use the `drives.list` method to
// retrieve the ID for a shared drive.
type Drive struct {
	// BackgroundImageFile: An image file and cropping parameters from which a
	// background image for this shared drive is set. This is a write only field;
	// it can only be set on `drive.drives.update` requests that don't set
	// `themeId`. When specified, all fields of the `backgroundImageFile` must be
	// set.
	BackgroundImageFile *DriveBackgroundImageFile `json:"backgroundImageFile,omitempty"`
	// BackgroundImageLink: Output only. A short-lived link to this shared drive's
	// background image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// Capabilities: Output only. Capabilities the current user has on this shared
	// drive.
	Capabilities *DriveCapabilities `json:"capabilities,omitempty"`
	// ColorRgb: The color of this shared drive as an RGB hex string. It can only
	// be set on a `drive.drives.update` request that does not set `themeId`.
	ColorRgb string `json:"colorRgb,omitempty"`
	// CreatedTime: The time at which the shared drive was created (RFC 3339
	// date-time).
	CreatedTime string `json:"createdTime,omitempty"`
	// Hidden: Whether the shared drive is hidden from default view.
	Hidden bool `json:"hidden,omitempty"`
	// Id: Output only. The ID of this shared drive which is also the ID of the top
	// level folder of this shared drive.
	Id string `json:"id,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#drive".
	Kind string `json:"kind,omitempty"`
	// Name: The name of this shared drive.
	Name string `json:"name,omitempty"`
	// OrgUnitId: Output only. The organizational unit of this shared drive. This
	// field is only populated on `drives.list` responses when the
	// `useDomainAdminAccess` parameter is set to `true`.
	OrgUnitId string `json:"orgUnitId,omitempty"`
	// Restrictions: A set of restrictions that apply to this shared drive or items
	// inside this shared drive. Note that restrictions can't be set when creating
	// a shared drive. To add a restriction, first create a shared drive and then
	// use `drives.update` to add restrictions.
	Restrictions *DriveRestrictions `json:"restrictions,omitempty"`
	// ThemeId: The ID of the theme from which the background image and color will
	// be set. The set of possible `driveThemes` can be retrieved from a
	// `drive.about.get` response. When not specified on a `drive.drives.create`
	// request, a random theme is chosen from which the background image and color
	// are set. This is a write-only field; it can only be set on requests that
	// don't set `colorRgb` or `backgroundImageFile`.
	ThemeId string `json:"themeId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageFile") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageFile") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Drive) MarshalJSON() ([]byte, error) {
	type NoMethod Drive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveBackgroundImageFile: An image file and cropping parameters from which a
// background image for this shared drive is set. This is a write only field;
// it can only be set on `drive.drives.update` requests that don't set
// `themeId`. When specified, all fields of the `backgroundImageFile` must be
// set.
type DriveBackgroundImageFile struct {
	// Id: The ID of an image file in Google Drive to use for the background image.
	Id string `json:"id,omitempty"`
	// Width: The width of the cropped image in the closed range of 0 to 1. This
	// value represents the width of the cropped image divided by the width of the
	// entire image. The height is computed by applying a width to height aspect
	// ratio of 80 to 9. The resulting image must be at least 1280 pixels wide and
	// 144 pixels high.
	Width float64 `json:"width,omitempty"`
	// XCoordinate: The X coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the horizontal distance from the left side of the entire
	// image to the left side of the cropping area divided by the width of the
	// entire image.
	XCoordinate float64 `json:"xCoordinate,omitempty"`
	// YCoordinate: The Y coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the vertical distance from the top side of the entire image
	// to the top side of the cropping area divided by the height of the entire
	// image.
	YCoordinate float64 `json:"yCoordinate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveBackgroundImageFile) MarshalJSON() ([]byte, error) {
	type NoMethod DriveBackgroundImageFile
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *DriveBackgroundImageFile) UnmarshalJSON(data []byte) error {
	type NoMethod DriveBackgroundImageFile
	var s1 struct {
		Width       gensupport.JSONFloat64 `json:"width"`
		XCoordinate gensupport.JSONFloat64 `json:"xCoordinate"`
		YCoordinate gensupport.JSONFloat64 `json:"yCoordinate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Width = float64(s1.Width)
	s.XCoordinate = float64(s1.XCoordinate)
	s.YCoordinate = float64(s1.YCoordinate)
	return nil
}

// DriveCapabilities: Output only. Capabilities the current user has on this
// shared drive.
type DriveCapabilities struct {
	// CanAddChildren: Output only. Whether the current user can add children to
	// folders in this shared drive.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanChangeCopyRequiresWriterPermissionRestriction: Output only. Whether the
	// current user can change the `copyRequiresWriterPermission` restriction of
	// this shared drive.
	CanChangeCopyRequiresWriterPermissionRestriction bool `json:"canChangeCopyRequiresWriterPermissionRestriction,omitempty"`
	// CanChangeDomainUsersOnlyRestriction: Output only. Whether the current user
	// can change the `domainUsersOnly` restriction of this shared drive.
	CanChangeDomainUsersOnlyRestriction bool `json:"canChangeDomainUsersOnlyRestriction,omitempty"`
	// CanChangeDownloadRestriction: Output only. Whether the current user can
	// change organizer-applied download restrictions of this shared drive.
	CanChangeDownloadRestriction bool `json:"canChangeDownloadRestriction,omitempty"`
	// CanChangeDriveBackground: Output only. Whether the current user can change
	// the background of this shared drive.
	CanChangeDriveBackground bool `json:"canChangeDriveBackground,omitempty"`
	// CanChangeDriveMembersOnlyRestriction: Output only. Whether the current user
	// can change the `driveMembersOnly` restriction of this shared drive.
	CanChangeDriveMembersOnlyRestriction bool `json:"canChangeDriveMembersOnlyRestriction,omitempty"`
	// CanChangeSharingFoldersRequiresOrganizerPermissionRestriction: Output only.
	// Whether the current user can change the
	// `sharingFoldersRequiresOrganizerPermission` restriction of this shared
	// drive.
	CanChangeSharingFoldersRequiresOrganizerPermissionRestriction bool `json:"canChangeSharingFoldersRequiresOrganizerPermissionRestriction,omitempty"`
	// CanComment: Output only. Whether the current user can comment on files in
	// this shared drive.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Output only. Whether the current user can copy files in this shared
	// drive.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDeleteChildren: Output only. Whether the current user can delete children
	// from folders in this shared drive.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDeleteDrive: Output only. Whether the current user can delete this shared
	// drive. Attempting to delete the shared drive may still fail if there are
	// untrashed items inside the shared drive.
	CanDeleteDrive bool `json:"canDeleteDrive,omitempty"`
	// CanDownload: Output only. Whether the current user can download files in
	// this shared drive.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Output only. Whether the current user can edit files in this shared
	// drive
	CanEdit bool `json:"canEdit,omitempty"`
	// CanListChildren: Output only. Whether the current user can list the children
	// of folders in this shared drive.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanManageMembers: Output only. Whether the current user can add members to
	// this shared drive or remove them or change their role.
	CanManageMembers bool `json:"canManageMembers,omitempty"`
	// CanReadRevisions: Output only. Whether the current user can read the
	// revisions resource of files in this shared drive.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanRename: Output only. Whether the current user can rename files or folders
	// in this shared drive.
	CanRename bool `json:"canRename,omitempty"`
	// CanRenameDrive: Output only. Whether the current user can rename this shared
	// drive.
	CanRenameDrive bool `json:"canRenameDrive,omitempty"`
	// CanResetDriveRestrictions: Output only. Whether the current user can reset
	// the shared drive restrictions to defaults.
	CanResetDriveRestrictions bool `json:"canResetDriveRestrictions,omitempty"`
	// CanShare: Output only. Whether the current user can share files or folders
	// in this shared drive.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrashChildren: Output only. Whether the current user can trash children
	// from folders in this shared drive.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAddChildren") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAddChildren") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod DriveCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveRestrictions: A set of restrictions that apply to this shared drive or
// items inside this shared drive. Note that restrictions can't be set when
// creating a shared drive. To add a restriction, first create a shared drive
// and then use `drives.update` to add restrictions.
type DriveRestrictions struct {
	// AdminManagedRestrictions: Whether administrative privileges on this shared
	// drive are required to modify restrictions.
	AdminManagedRestrictions bool `json:"adminManagedRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download files inside this shared drive, should be disabled for readers and
	// commenters. When this restriction is set to `true`, it will override the
	// similarly named field to `true` for any file inside this shared drive.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// DomainUsersOnly: Whether access to this shared drive and items inside this
	// shared drive is restricted to users of the domain to which this shared drive
	// belongs. This restriction may be overridden by other sharing policies
	// controlled outside of this shared drive.
	DomainUsersOnly bool `json:"domainUsersOnly,omitempty"`
	// DownloadRestriction: Download restrictions applied by shared drive managers.
	DownloadRestriction *DownloadRestriction `json:"downloadRestriction,omitempty"`
	// DriveMembersOnly: Whether access to items inside this shared drive is
	// restricted to its members.
	DriveMembersOnly bool `json:"driveMembersOnly,omitempty"`
	// SharingFoldersRequiresOrganizerPermission: If true, only users with the
	// organizer role can share folders. If false, users with either the organizer
	// role or the file organizer role can share folders.
	SharingFoldersRequiresOrganizerPermission bool `json:"sharingFoldersRequiresOrganizerPermission,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdminManagedRestrictions")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdminManagedRestrictions") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveRestrictions) MarshalJSON() ([]byte, error) {
	type NoMethod DriveRestrictions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveList: A list of shared drives.
type DriveList struct {
	// Drives: The list of shared drives. If nextPageToken is populated, then this
	// list may be incomplete and an additional page of results should be fetched.
	Drives []*Drive `json:"drives,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#driveList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of shared drives. This will
	// be absent if the end of the list has been reached. If the token is rejected
	// for any reason, it should be discarded, and pagination should be restarted
	// from the first page of results. The page token is typically valid for
	// several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Drives") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Drives") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveList) MarshalJSON() ([]byte, error) {
	type NoMethod DriveList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// File: The metadata for a file. Some resource methods (such as
// `files.update`) require a `fileId`. Use the `files.list` method to retrieve
// the ID for a file.
type File struct {
	// AppProperties: A collection of arbitrary key-value pairs which are private
	// to the requesting app.
	// Entries with null values are cleared in update and copy requests. These
	// properties can only be retrieved using an authenticated request. An
	// authenticated request uses an access token obtained with a OAuth 2 client
	// ID. You cannot use an API key to retrieve private properties.
	AppProperties map[string]string `json:"appProperties,omitempty"`
	// Capabilities: Output only. Capabilities the current user has on this file.
	// Each capability corresponds to a fine-grained action that a user may take.
	// For more information, see Understand file capabilities
	// (https://developers.google.com/workspace/drive/api/guides/manage-sharing#capabilities).
	Capabilities *FileCapabilities `json:"capabilities,omitempty"`
	// ContentHints: Additional information about the content of the file. These
	// fields are never populated in responses.
	ContentHints *FileContentHints `json:"contentHints,omitempty"`
	// ContentRestrictions: Restrictions for accessing the content of the file.
	// Only populated if such a restriction exists.
	ContentRestrictions []*ContentRestriction `json:"contentRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download this file should be disabled for readers and commenters.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// CreatedTime: The time at which the file was created (RFC 3339 date-time).
	CreatedTime string `json:"createdTime,omitempty"`
	// Description: A short description of the file.
	Description string `json:"description,omitempty"`
	// DownloadRestrictions: Download restrictions applied on the file.
	DownloadRestrictions *DownloadRestrictionsMetadata `json:"downloadRestrictions,omitempty"`
	// DriveId: Output only. ID of the shared drive the file resides in. Only
	// populated for items in shared drives.
	DriveId string `json:"driveId,omitempty"`
	// ExplicitlyTrashed: Output only. Whether the file has been explicitly
	// trashed, as opposed to recursively trashed from a parent folder.
	ExplicitlyTrashed bool `json:"explicitlyTrashed,omitempty"`
	// ExportLinks: Output only. Links for exporting Docs Editors files to specific
	// formats.
	ExportLinks map[string]string `json:"exportLinks,omitempty"`
	// FileExtension: Output only. The final component of `fullFileExtension`. This
	// is only available for files with binary content in Google Drive.
	FileExtension string `json:"fileExtension,omitempty"`
	// FolderColorRgb: The color for a folder or a shortcut to a folder as an RGB
	// hex string. The supported colors are published in the `folderColorPalette`
	// field of the `about` (/workspace/drive/api/reference/rest/v3/about)
	// resource. If an unsupported color is specified, the closest color in the
	// palette is used instead.
	FolderColorRgb string `json:"folderColorRgb,omitempty"`
	// FullFileExtension: Output only. The full file extension extracted from the
	// `name` field. May contain multiple concatenated extensions, such as
	// "tar.gz". This is only available for files with binary content in Google
	// Drive. This is automatically updated when the `name` field changes, however
	// it's not cleared if the new name doesn't contain a valid extension.
	FullFileExtension string `json:"fullFileExtension,omitempty"`
	// HasAugmentedPermissions: Output only. Whether there are permissions directly
	// on this file. This field is only populated for items in shared drives.
	HasAugmentedPermissions bool `json:"hasAugmentedPermissions,omitempty"`
	// HasThumbnail: Output only. Whether this file has a thumbnail. This doesn't
	// indicate whether the requesting app has access to the thumbnail. To check
	// access, look for the presence of the thumbnailLink field.
	HasThumbnail bool `json:"hasThumbnail,omitempty"`
	// HeadRevisionId: Output only. The ID of the file's head revision. This is
	// currently only available for files with binary content in Google Drive.
	HeadRevisionId string `json:"headRevisionId,omitempty"`
	// IconLink: Output only. A static, unauthenticated link to the file's icon.
	IconLink string `json:"iconLink,omitempty"`
	// Id: The ID of the file.
	Id string `json:"id,omitempty"`
	// ImageMediaMetadata: Output only. Additional metadata about image media, if
	// available.
	ImageMediaMetadata *FileImageMediaMetadata `json:"imageMediaMetadata,omitempty"`
	// InheritedPermissionsDisabled: Whether this file has inherited permissions
	// disabled. Inherited permissions are enabled by default.
	InheritedPermissionsDisabled bool `json:"inheritedPermissionsDisabled,omitempty"`
	// IsAppAuthorized: Output only. Whether the file was created or opened by the
	// requesting app.
	IsAppAuthorized bool `json:"isAppAuthorized,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#file".
	Kind string `json:"kind,omitempty"`
	// LabelInfo: Output only. An overview of the labels on the file.
	LabelInfo *FileLabelInfo `json:"labelInfo,omitempty"`
	// LastModifyingUser: Output only. The last user to modify the file. This field
	// is only populated when the last modification was performed by a signed-in
	// user.
	LastModifyingUser *User `json:"lastModifyingUser,omitempty"`
	// LinkShareMetadata: Contains details about the link URLs that clients are
	// using to refer to this item.
	LinkShareMetadata *FileLinkShareMetadata `json:"linkShareMetadata,omitempty"`
	// Md5Checksum: Output only. The MD5 checksum for the content of the file. This
	// is only applicable to files with binary content in Google Drive.
	Md5Checksum string `json:"md5Checksum,omitempty"`
	// MimeType: The MIME type of the file. Google Drive attempts to automatically
	// detect an appropriate value from uploaded content, if no value is provided.
	// The value cannot be changed unless a new revision is uploaded. If a file is
	// created with a Google Doc MIME type, the uploaded content is imported, if
	// possible. The supported import formats are published in the `about`
	// (/workspace/drive/api/reference/rest/v3/about) resource.
	MimeType string `json:"mimeType,omitempty"`
	// ModifiedByMe: Output only. Whether the file has been modified by this user.
	ModifiedByMe bool `json:"modifiedByMe,omitempty"`
	// ModifiedByMeTime: The last time the file was modified by the user (RFC 3339
	// date-time).
	ModifiedByMeTime string `json:"modifiedByMeTime,omitempty"`
	// ModifiedTime: he last time the file was modified by anyone (RFC 3339
	// date-time). Note that setting modifiedTime will also update modifiedByMeTime
	// for the user.
	ModifiedTime string `json:"modifiedTime,omitempty"`
	// Name: The name of the file. This isn't necessarily unique within a folder.
	// Note that for immutable items such as the top-level folders of shared
	// drives, the My Drive root folder, and the Application Data folder, the name
	// is constant.
	Name string `json:"name,omitempty"`
	// OriginalFilename: The original filename of the uploaded content if
	// available, or else the original value of the `name` field. This is only
	// available for files with binary content in Google Drive.
	OriginalFilename string `json:"originalFilename,omitempty"`
	// OwnedByMe: Output only. Whether the user owns the file. Not populated for
	// items in shared drives.
	OwnedByMe bool `json:"ownedByMe,omitempty"`
	// Owners: Output only. The owner of this file. Only certain legacy files may
	// have more than one owner. This field isn't populated for items in shared
	// drives.
	Owners []*User `json:"owners,omitempty"`
	// Parents: The ID of the parent folder containing the file. A file can only
	// have one parent folder; specifying multiple parents isn't supported. If not
	// specified as part of a create request, the file is placed directly in the
	// user's My Drive folder. If not specified as part of a copy request, the file
	// inherits any discoverable parent of the source file. Update requests must
	// use the `addParents` and `removeParents` parameters to modify the parents
	// list.
	Parents []string `json:"parents,omitempty"`
	// PermissionIds: Output only. List of permission IDs for users with access to
	// this file.
	PermissionIds []string `json:"permissionIds,omitempty"`
	// Permissions: Output only. The full list of permissions for the file. This is
	// only available if the requesting user can share the file. Not populated for
	// items in shared drives.
	Permissions []*Permission `json:"permissions,omitempty"`
	// Properties: A collection of arbitrary key-value pairs which are visible to
	// all apps.
	// Entries with null values are cleared in update and copy requests.
	Properties map[string]string `json:"properties,omitempty"`
	// QuotaBytesUsed: Output only. The number of storage quota bytes used by the
	// file. This includes the head revision as well as previous revisions with
	// `keepForever` enabled.
	QuotaBytesUsed int64 `json:"quotaBytesUsed,omitempty,string"`
	// ResourceKey: Output only. A key needed to access the item via a shared link.
	ResourceKey string `json:"resourceKey,omitempty"`
	// Sha1Checksum: Output only. The SHA1 checksum associated with this file, if
	// available. This field is only populated for files with content stored in
	// Google Drive; it's not populated for Docs Editors or shortcut files.
	Sha1Checksum string `json:"sha1Checksum,omitempty"`
	// Sha256Checksum: Output only. The SHA256 checksum associated with this file,
	// if available. This field is only populated for files with content stored in
	// Google Drive; it's not populated for Docs Editors or shortcut files.
	Sha256Checksum string `json:"sha256Checksum,omitempty"`
	// Shared: Output only. Whether the file has been shared. Not populated for
	// items in shared drives.
	Shared bool `json:"shared,omitempty"`
	// SharedWithMeTime: The time at which the file was shared with the user, if
	// applicable (RFC 3339 date-time).
	SharedWithMeTime string `json:"sharedWithMeTime,omitempty"`
	// SharingUser: Output only. The user who shared the file with the requesting
	// user, if applicable.
	SharingUser *User `json:"sharingUser,omitempty"`
	// ShortcutDetails: Shortcut file details. Only populated for shortcut files,
	// which have the mimeType field set to `application/vnd.google-apps.shortcut`.
	// Can only be set on `files.create` requests.
	ShortcutDetails *FileShortcutDetails `json:"shortcutDetails,omitempty"`
	// Size: Output only. Size in bytes of blobs and Google Workspace editor files.
	// Won't be populated for files that have no size, like shortcuts and folders.
	Size int64 `json:"size,omitempty,string"`
	// Spaces: Output only. The list of spaces which contain the file. The
	// currently supported values are `drive`, `appDataFolder`, and `photos`.
	Spaces []string `json:"spaces,omitempty"`
	// Starred: Whether the user has starred the file.
	Starred bool `json:"starred,omitempty"`
	// TeamDriveId: Deprecated: Output only. Use `driveId` instead.
	TeamDriveId string `json:"teamDriveId,omitempty"`
	// ThumbnailLink: Output only. A short-lived link to the file's thumbnail, if
	// available. Typically lasts on the order of hours. Not intended for direct
	// usage on web applications due to Cross-Origin Resource Sharing (CORS)
	// (https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS) policies. Consider
	// using a proxy server. Only populated when the requesting app can access the
	// file's content. If the file isn't shared publicly, the URL returned in
	// `files.thumbnailLink` must be fetched using a credentialed request.
	ThumbnailLink string `json:"thumbnailLink,omitempty"`
	// ThumbnailVersion: Output only. The thumbnail version for use in thumbnail
	// cache invalidation.
	ThumbnailVersion int64 `json:"thumbnailVersion,omitempty,string"`
	// Trashed: Whether the file has been trashed, either explicitly or from a
	// trashed parent folder. Only the owner may trash a file, and other users
	// cannot see files in the owner's trash.
	Trashed bool `json:"trashed,omitempty"`
	// TrashedTime: The time that the item was trashed (RFC 3339 date-time). Only
	// populated for items in shared drives.
	TrashedTime string `json:"trashedTime,omitempty"`
	// TrashingUser: Output only. If the file has been explicitly trashed, the user
	// who trashed it. Only populated for items in shared drives.
	TrashingUser *User `json:"trashingUser,omitempty"`
	// Version: Output only. A monotonically increasing version number for the
	// file. This reflects every change made to the file on the server, even those
	// not visible to the user.
	Version int64 `json:"version,omitempty,string"`
	// VideoMediaMetadata: Output only. Additional metadata about video media. This
	// may not be available immediately upon upload.
	VideoMediaMetadata *FileVideoMediaMetadata `json:"videoMediaMetadata,omitempty"`
	// ViewedByMe: Output only. Whether the file has been viewed by this user.
	ViewedByMe bool `json:"viewedByMe,omitempty"`
	// ViewedByMeTime: The last time the file was viewed by the user (RFC 3339
	// date-time).
	ViewedByMeTime string `json:"viewedByMeTime,omitempty"`
	// ViewersCanCopyContent: Deprecated: Use `copyRequiresWriterPermission`
	// instead.
	ViewersCanCopyContent bool `json:"viewersCanCopyContent,omitempty"`
	// WebContentLink: Output only. A link for downloading the content of the file
	// in a browser. This is only available for files with binary content in Google
	// Drive.
	WebContentLink string `json:"webContentLink,omitempty"`
	// WebViewLink: Output only. A link for opening the file in a relevant Google
	// editor or viewer in a browser.
	WebViewLink string `json:"webViewLink,omitempty"`
	// WritersCanShare: Whether users with only `writer` permission can modify the
	// file's permissions. Not populated for items in shared drives.
	WritersCanShare bool `json:"writersCanShare,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AppProperties") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AppProperties") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s File) MarshalJSON() ([]byte, error) {
	type NoMethod File
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileCapabilities: Output only. Capabilities the current user has on this
// file. Each capability corresponds to a fine-grained action that a user may
// take. For more information, see Understand file capabilities
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing#capabilities).
type FileCapabilities struct {
	// CanAcceptOwnership: Output only. Whether the current user is the pending
	// owner of the file. Not populated for shared drive files.
	CanAcceptOwnership bool `json:"canAcceptOwnership,omitempty"`
	// CanAddChildren: Output only. Whether the current user can add children to
	// this folder. This is always `false` when the item isn't a folder.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanAddFolderFromAnotherDrive: Output only. Whether the current user can add
	// a folder from another drive (different shared drive or My Drive) to this
	// folder. This is `false` when the item isn't a folder. Only populated for
	// items in shared drives.
	CanAddFolderFromAnotherDrive bool `json:"canAddFolderFromAnotherDrive,omitempty"`
	// CanAddMyDriveParent: Output only. Whether the current user can add a parent
	// for the item without removing an existing parent in the same request. Not
	// populated for shared drive files.
	CanAddMyDriveParent bool `json:"canAddMyDriveParent,omitempty"`
	// CanChangeCopyRequiresWriterPermission: Output only. Whether the current user
	// can change the `copyRequiresWriterPermission` restriction of this file.
	CanChangeCopyRequiresWriterPermission bool `json:"canChangeCopyRequiresWriterPermission,omitempty"`
	// CanChangeItemDownloadRestriction: Output only. Whether the current user can
	// change the owner or organizer-applied download restrictions of the file.
	CanChangeItemDownloadRestriction bool `json:"canChangeItemDownloadRestriction,omitempty"`
	// CanChangeSecurityUpdateEnabled: Output only. Whether the current user can
	// change the `securityUpdateEnabled` field on link share metadata.
	CanChangeSecurityUpdateEnabled bool `json:"canChangeSecurityUpdateEnabled,omitempty"`
	// CanChangeViewersCanCopyContent: Deprecated: Output only.
	CanChangeViewersCanCopyContent bool `json:"canChangeViewersCanCopyContent,omitempty"`
	// CanComment: Output only. Whether the current user can comment on this file.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Output only. Whether the current user can copy this file. For an
	// item in a shared drive, whether the current user can copy non-folder
	// descendants of this item, or this item if it's not a folder.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDelete: Output only. Whether the current user can delete this file.
	CanDelete bool `json:"canDelete,omitempty"`
	// CanDeleteChildren: Output only. Whether the current user can delete children
	// of this folder. This is `false` when the item isn't a folder. Only populated
	// for items in shared drives.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDisableInheritedPermissions: Whether a user can disable inherited
	// permissions.
	CanDisableInheritedPermissions bool `json:"canDisableInheritedPermissions,omitempty"`
	// CanDownload: Output only. Whether the current user can download this file.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Output only. Whether the current user can edit this file. Other
	// factors may limit the type of changes a user can make to a file. For
	// example, see `canChangeCopyRequiresWriterPermission` or `canModifyContent`.
	CanEdit bool `json:"canEdit,omitempty"`
	// CanEnableInheritedPermissions: Whether a user can re-enable inherited
	// permissions.
	CanEnableInheritedPermissions bool `json:"canEnableInheritedPermissions,omitempty"`
	// CanListChildren: Output only. Whether the current user can list the children
	// of this folder. This is always `false` when the item isn't a folder.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanModifyContent: Output only. Whether the current user can modify the
	// content of this file.
	CanModifyContent bool `json:"canModifyContent,omitempty"`
	// CanModifyContentRestriction: Deprecated: Output only. Use one of
	// `canModifyEditorContentRestriction`, `canModifyOwnerContentRestriction`, or
	// `canRemoveContentRestriction`.
	CanModifyContentRestriction bool `json:"canModifyContentRestriction,omitempty"`
	// CanModifyEditorContentRestriction: Output only. Whether the current user can
	// add or modify content restrictions on the file which are editor restricted.
	CanModifyEditorContentRestriction bool `json:"canModifyEditorContentRestriction,omitempty"`
	// CanModifyLabels: Output only. Whether the current user can modify the labels
	// on the file.
	CanModifyLabels bool `json:"canModifyLabels,omitempty"`
	// CanModifyOwnerContentRestriction: Output only. Whether the current user can
	// add or modify content restrictions which are owner restricted.
	CanModifyOwnerContentRestriction bool `json:"canModifyOwnerContentRestriction,omitempty"`
	// CanMoveChildrenOutOfDrive: Output only. Whether the current user can move
	// children of this folder outside of the shared drive. This is `false` when
	// the item isn't a folder. Only populated for items in shared drives.
	CanMoveChildrenOutOfDrive bool `json:"canMoveChildrenOutOfDrive,omitempty"`
	// CanMoveChildrenOutOfTeamDrive: Deprecated: Output only. Use
	// `canMoveChildrenOutOfDrive` instead.
	CanMoveChildrenOutOfTeamDrive bool `json:"canMoveChildrenOutOfTeamDrive,omitempty"`
	// CanMoveChildrenWithinDrive: Output only. Whether the current user can move
	// children of this folder within this drive. This is `false` when the item
	// isn't a folder. Note that a request to move the child may still fail
	// depending on the current user's access to the child and to the destination
	// folder.
	CanMoveChildrenWithinDrive bool `json:"canMoveChildrenWithinDrive,omitempty"`
	// CanMoveChildrenWithinTeamDrive: Deprecated: Output only. Use
	// `canMoveChildrenWithinDrive` instead.
	CanMoveChildrenWithinTeamDrive bool `json:"canMoveChildrenWithinTeamDrive,omitempty"`
	// CanMoveItemIntoTeamDrive: Deprecated: Output only. Use
	// `canMoveItemOutOfDrive` instead.
	CanMoveItemIntoTeamDrive bool `json:"canMoveItemIntoTeamDrive,omitempty"`
	// CanMoveItemOutOfDrive: Output only. Whether the current user can move this
	// item outside of this drive by changing its parent. Note that a request to
	// change the parent of the item may still fail depending on the new parent
	// that's being added.
	CanMoveItemOutOfDrive bool `json:"canMoveItemOutOfDrive,omitempty"`
	// CanMoveItemOutOfTeamDrive: Deprecated: Output only. Use
	// `canMoveItemOutOfDrive` instead.
	CanMoveItemOutOfTeamDrive bool `json:"canMoveItemOutOfTeamDrive,omitempty"`
	// CanMoveItemWithinDrive: Output only. Whether the current user can move this
	// item within this drive. Note that a request to change the parent of the item
	// may still fail depending on the new parent that's being added and the parent
	// that is being removed.
	CanMoveItemWithinDrive bool `json:"canMoveItemWithinDrive,omitempty"`
	// CanMoveItemWithinTeamDrive: Deprecated: Output only. Use
	// `canMoveItemWithinDrive` instead.
	CanMoveItemWithinTeamDrive bool `json:"canMoveItemWithinTeamDrive,omitempty"`
	// CanMoveTeamDriveItem: Deprecated: Output only. Use `canMoveItemWithinDrive`
	// or `canMoveItemOutOfDrive` instead.
	CanMoveTeamDriveItem bool `json:"canMoveTeamDriveItem,omitempty"`
	// CanReadDrive: Output only. Whether the current user can read the shared
	// drive to which this file belongs. Only populated for items in shared drives.
	CanReadDrive bool `json:"canReadDrive,omitempty"`
	// CanReadLabels: Output only. Whether the current user can read the labels on
	// the file.
	CanReadLabels bool `json:"canReadLabels,omitempty"`
	// CanReadRevisions: Output only. Whether the current user can read the
	// revisions resource of this file. For a shared drive item, whether revisions
	// of non-folder descendants of this item, or this item if it's not a folder,
	// can be read.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanReadTeamDrive: Deprecated: Output only. Use `canReadDrive` instead.
	CanReadTeamDrive bool `json:"canReadTeamDrive,omitempty"`
	// CanRemoveChildren: Output only. Whether the current user can remove children
	// from this folder. This is always `false` when the item isn't a folder. For a
	// folder in a shared drive, use `canDeleteChildren` or `canTrashChildren`
	// instead.
	CanRemoveChildren bool `json:"canRemoveChildren,omitempty"`
	// CanRemoveContentRestriction: Output only. Whether there's a content
	// restriction on the file that can be removed by the current user.
	CanRemoveContentRestriction bool `json:"canRemoveContentRestriction,omitempty"`
	// CanRemoveMyDriveParent: Output only. Whether the current user can remove a
	// parent from the item without adding another parent in the same request. Not
	// populated for shared drive files.
	CanRemoveMyDriveParent bool `json:"canRemoveMyDriveParent,omitempty"`
	// CanRename: Output only. Whether the current user can rename this file.
	CanRename bool `json:"canRename,omitempty"`
	// CanShare: Output only. Whether the current user can modify the sharing
	// settings for this file.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrash: Output only. Whether the current user can move this file to trash.
	CanTrash bool `json:"canTrash,omitempty"`
	// CanTrashChildren: Output only. Whether the current user can trash children
	// of this folder. This is `false` when the item isn't a folder. Only populated
	// for items in shared drives.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// CanUntrash: Output only. Whether the current user can restore this file from
	// trash.
	CanUntrash bool `json:"canUntrash,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAcceptOwnership") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAcceptOwnership") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod FileCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileContentHints: Additional information about the content of the file.
// These fields are never populated in responses.
type FileContentHints struct {
	// IndexableText: Text to be indexed for the file to improve fullText queries.
	// This is limited to 128 KB in length and may contain HTML elements.
	IndexableText string `json:"indexableText,omitempty"`
	// Thumbnail: A thumbnail for the file. This will only be used if Google Drive
	// cannot generate a standard thumbnail.
	Thumbnail *FileContentHintsThumbnail `json:"thumbnail,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IndexableText") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IndexableText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileContentHints) MarshalJSON() ([]byte, error) {
	type NoMethod FileContentHints
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileContentHintsThumbnail: A thumbnail for the file. This will only be used
// if Google Drive cannot generate a standard thumbnail.
type FileContentHintsThumbnail struct {
	// Image: The thumbnail data encoded with URL-safe Base64 (RFC 4648 section 5
	// (https://datatracker.ietf.org/doc/html/rfc4648#section-5)).
	Image string `json:"image,omitempty"`
	// MimeType: The MIME type of the thumbnail.
	MimeType string `json:"mimeType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Image") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Image") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileContentHintsThumbnail) MarshalJSON() ([]byte, error) {
	type NoMethod FileContentHintsThumbnail
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileImageMediaMetadata: Output only. Additional metadata about image media,
// if available.
type FileImageMediaMetadata struct {
	// Aperture: Output only. The aperture used to create the photo (f-number).
	Aperture float64 `json:"aperture,omitempty"`
	// CameraMake: Output only. The make of the camera used to create the photo.
	CameraMake string `json:"cameraMake,omitempty"`
	// CameraModel: Output only. The model of the camera used to create the photo.
	CameraModel string `json:"cameraModel,omitempty"`
	// ColorSpace: Output only. The color space of the photo.
	ColorSpace string `json:"colorSpace,omitempty"`
	// ExposureBias: Output only. The exposure bias of the photo (APEX value).
	ExposureBias float64 `json:"exposureBias,omitempty"`
	// ExposureMode: Output only. The exposure mode used to create the photo.
	ExposureMode string `json:"exposureMode,omitempty"`
	// ExposureTime: Output only. The length of the exposure, in seconds.
	ExposureTime float64 `json:"exposureTime,omitempty"`
	// FlashUsed: Output only. Whether a flash was used to create the photo.
	FlashUsed bool `json:"flashUsed,omitempty"`
	// FocalLength: Output only. The focal length used to create the photo, in
	// millimeters.
	FocalLength float64 `json:"focalLength,omitempty"`
	// Height: Output only. The height of the image in pixels.
	Height int64 `json:"height,omitempty"`
	// IsoSpeed: Output only. The ISO speed used to create the photo.
	IsoSpeed int64 `json:"isoSpeed,omitempty"`
	// Lens: Output only. The lens used to create the photo.
	Lens string `json:"lens,omitempty"`
	// Location: Output only. Geographic location information stored in the image.
	Location *FileImageMediaMetadataLocation `json:"location,omitempty"`
	// MaxApertureValue: Output only. The smallest f-number of the lens at the
	// focal length used to create the photo (APEX value).
	MaxApertureValue float64 `json:"maxApertureValue,omitempty"`
	// MeteringMode: Output only. The metering mode used to create the photo.
	MeteringMode string `json:"meteringMode,omitempty"`
	// Rotation: Output only. The number of clockwise 90 degree rotations applied
	// from the image's original orientation.
	Rotation int64 `json:"rotation,omitempty"`
	// Sensor: Output only. The type of sensor used to create the photo.
	Sensor string `json:"sensor,omitempty"`
	// SubjectDistance: Output only. The distance to the subject of the photo, in
	// meters.
	SubjectDistance int64 `json:"subjectDistance,omitempty"`
	// Time: Output only. The date and time the photo was taken (EXIF DateTime).
	Time string `json:"time,omitempty"`
	// WhiteBalance: Output only. The white balance mode used to create the photo.
	WhiteBalance string `json:"whiteBalance,omitempty"`
	// Width: Output only. The width of the image in pixels.
	Width int64 `json:"width,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aperture") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aperture") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileImageMediaMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileImageMediaMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FileImageMediaMetadata) UnmarshalJSON(data []byte) error {
	type NoMethod FileImageMediaMetadata
	var s1 struct {
		Aperture         gensupport.JSONFloat64 `json:"aperture"`
		ExposureBias     gensupport.JSONFloat64 `json:"exposureBias"`
		ExposureTime     gensupport.JSONFloat64 `json:"exposureTime"`
		FocalLength      gensupport.JSONFloat64 `json:"focalLength"`
		MaxApertureValue gensupport.JSONFloat64 `json:"maxApertureValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Aperture = float64(s1.Aperture)
	s.ExposureBias = float64(s1.ExposureBias)
	s.ExposureTime = float64(s1.ExposureTime)
	s.FocalLength = float64(s1.FocalLength)
	s.MaxApertureValue = float64(s1.MaxApertureValue)
	return nil
}

// FileImageMediaMetadataLocation: Output only. Geographic location information
// stored in the image.
type FileImageMediaMetadataLocation struct {
	// Altitude: Output only. The altitude stored in the image.
	Altitude float64 `json:"altitude,omitempty"`
	// Latitude: Output only. The latitude stored in the image.
	Latitude float64 `json:"latitude,omitempty"`
	// Longitude: Output only. The longitude stored in the image.
	Longitude float64 `json:"longitude,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Altitude") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Altitude") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileImageMediaMetadataLocation) MarshalJSON() ([]byte, error) {
	type NoMethod FileImageMediaMetadataLocation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FileImageMediaMetadataLocation) UnmarshalJSON(data []byte) error {
	type NoMethod FileImageMediaMetadataLocation
	var s1 struct {
		Altitude  gensupport.JSONFloat64 `json:"altitude"`
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Altitude = float64(s1.Altitude)
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

// FileLabelInfo: Output only. An overview of the labels on the file.
type FileLabelInfo struct {
	// Labels: Output only. The set of labels on the file as requested by the label
	// IDs in the `includeLabels` parameter. By default, no labels are returned.
	Labels []*Label `json:"labels,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Labels") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Labels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileLabelInfo) MarshalJSON() ([]byte, error) {
	type NoMethod FileLabelInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileLinkShareMetadata: Contains details about the link URLs that clients are
// using to refer to this item.
type FileLinkShareMetadata struct {
	// SecurityUpdateEligible: Output only. Whether the file is eligible for
	// security update.
	SecurityUpdateEligible bool `json:"securityUpdateEligible,omitempty"`
	// SecurityUpdateEnabled: Output only. Whether the security update is enabled
	// for this file.
	SecurityUpdateEnabled bool `json:"securityUpdateEnabled,omitempty"`
	// ForceSendFields is a list of field names (e.g. "SecurityUpdateEligible") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SecurityUpdateEligible") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileLinkShareMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileLinkShareMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileShortcutDetails: Shortcut file details. Only populated for shortcut
// files, which have the mimeType field set to
// `application/vnd.google-apps.shortcut`. Can only be set on `files.create`
// requests.
type FileShortcutDetails struct {
	// TargetId: The ID of the file that this shortcut points to. Can only be set
	// on `files.create` requests.
	TargetId string `json:"targetId,omitempty"`
	// TargetMimeType: Output only. The MIME type of the file that this shortcut
	// points to. The value of this field is a snapshot of the target's MIME type,
	// captured when the shortcut is created.
	TargetMimeType string `json:"targetMimeType,omitempty"`
	// TargetResourceKey: Output only. The `resourceKey` for the target file.
	TargetResourceKey string `json:"targetResourceKey,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TargetId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TargetId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileShortcutDetails) MarshalJSON() ([]byte, error) {
	type NoMethod FileShortcutDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileVideoMediaMetadata: Output only. Additional metadata about video media.
// This may not be available immediately upon upload.
type FileVideoMediaMetadata struct {
	// DurationMillis: Output only. The duration of the video in milliseconds.
	DurationMillis int64 `json:"durationMillis,omitempty,string"`
	// Height: Output only. The height of the video in pixels.
	Height int64 `json:"height,omitempty"`
	// Width: Output only. The width of the video in pixels.
	Width int64 `json:"width,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DurationMillis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DurationMillis") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileVideoMediaMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileVideoMediaMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileList: A list of files.
type FileList struct {
	// Files: The list of files. If `nextPageToken` is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Files []*File `json:"files,omitempty"`
	// IncompleteSearch: Whether the search process was incomplete. If true, then
	// some search results might be missing, since all documents were not searched.
	// This can occur when searching multiple drives with the `allDrives` corpora,
	// but all corpora couldn't be searched. When this happens, it's suggested that
	// clients narrow their query by choosing a different corpus such as `user` or
	// `drive`.
	IncompleteSearch bool `json:"incompleteSearch,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#fileList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of files. This will be
	// absent if the end of the files list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results. The page token is typically valid
	// for several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Files") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Files") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileList) MarshalJSON() ([]byte, error) {
	type NoMethod FileList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GeneratedIds: A list of generated file IDs which can be provided in create
// requests.
type GeneratedIds struct {
	// Ids: The IDs generated for the requesting user in the specified space.
	Ids []string `json:"ids,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#generatedIds".
	Kind string `json:"kind,omitempty"`
	// Space: The type of file that can be created with these IDs.
	Space string `json:"space,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Ids") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Ids") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GeneratedIds) MarshalJSON() ([]byte, error) {
	type NoMethod GeneratedIds
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Label: Representation of label and label fields.
type Label struct {
	// Fields: A map of the fields on the label, keyed by the field's ID.
	Fields map[string]LabelField `json:"fields,omitempty"`
	// Id: The ID of the label.
	Id string `json:"id,omitempty"`
	// Kind: This is always drive#label
	Kind string `json:"kind,omitempty"`
	// RevisionId: The revision ID of the label.
	RevisionId string `json:"revisionId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Fields") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Fields") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Label) MarshalJSON() ([]byte, error) {
	type NoMethod Label
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelField: Representation of field, which is a typed key-value pair.
type LabelField struct {
	// DateString: Only present if valueType is dateString. RFC 3339 formatted
	// date: YYYY-MM-DD.
	DateString []string `json:"dateString,omitempty"`
	// Id: The identifier of this label field.
	Id string `json:"id,omitempty"`
	// Integer: Only present if `valueType` is `integer`.
	Integer googleapi.Int64s `json:"integer,omitempty"`
	// Kind: This is always drive#labelField.
	Kind string `json:"kind,omitempty"`
	// Selection: Only present if `valueType` is `selection`
	Selection []string `json:"selection,omitempty"`
	// Text: Only present if `valueType` is `text`.
	Text []string `json:"text,omitempty"`
	// User: Only present if `valueType` is `user`.
	User []*User `json:"user,omitempty"`
	// ValueType: The field type. While new values may be supported in the future,
	// the following are currently allowed: * `dateString` * `integer` *
	// `selection` * `text` * `user`
	ValueType string `json:"valueType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateString") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateString") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelField) MarshalJSON() ([]byte, error) {
	type NoMethod LabelField
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelFieldModification: A modification to a label's field.
type LabelFieldModification struct {
	// FieldId: The ID of the field to be modified.
	FieldId string `json:"fieldId,omitempty"`
	// Kind: This is always "drive#labelFieldModification".
	Kind string `json:"kind,omitempty"`
	// SetDateValues: Replaces the value of a dateString Field with these new
	// values. The string must be in the RFC 3339 full-date format: YYYY-MM-DD.
	SetDateValues []string `json:"setDateValues,omitempty"`
	// SetIntegerValues: Replaces the value of an `integer` field with these new
	// values.
	SetIntegerValues googleapi.Int64s `json:"setIntegerValues,omitempty"`
	// SetSelectionValues: Replaces a `selection` field with these new values.
	SetSelectionValues []string `json:"setSelectionValues,omitempty"`
	// SetTextValues: Sets the value of a `text` field.
	SetTextValues []string `json:"setTextValues,omitempty"`
	// SetUserValues: Replaces a `user` field with these new values. The values
	// must be a valid email addresses.
	SetUserValues []string `json:"setUserValues,omitempty"`
	// UnsetValues: Unsets the values for this field.
	UnsetValues bool `json:"unsetValues,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FieldId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FieldId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelFieldModification) MarshalJSON() ([]byte, error) {
	type NoMethod LabelFieldModification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelList: A list of labels applied to a file.
type LabelList struct {
	// Kind: This is always "drive#labelList".
	Kind string `json:"kind,omitempty"`
	// Labels: The list of labels.
	Labels []*Label `json:"labels,omitempty"`
	// NextPageToken: The page token for the next page of labels. This field will
	// be absent if the end of the list has been reached. If the token is rejected
	// for any reason, it should be discarded, and pagination should be restarted
	// from the first page of results. The page token is typically valid for
	// several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelList) MarshalJSON() ([]byte, error) {
	type NoMethod LabelList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelModification: A modification to a label on a file. A
// `LabelModification` can be used to apply a label to a file, update an
// existing label on a file, or remove a label from a file.
type LabelModification struct {
	// FieldModifications: The list of modifications to this label's fields.
	FieldModifications []*LabelFieldModification `json:"fieldModifications,omitempty"`
	// Kind: This is always "drive#labelModification".
	Kind string `json:"kind,omitempty"`
	// LabelId: The ID of the label to modify.
	LabelId string `json:"labelId,omitempty"`
	// RemoveLabel: If true, the label will be removed from the file.
	RemoveLabel bool `json:"removeLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FieldModifications") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FieldModifications") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelModification) MarshalJSON() ([]byte, error) {
	type NoMethod LabelModification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccessProposalsResponse: The response to an access proposal list
// request.
type ListAccessProposalsResponse struct {
	// AccessProposals: The list of access proposals. This field is only populated
	// in Drive API v3.
	AccessProposals []*AccessProposal `json:"accessProposals,omitempty"`
	// NextPageToken: The continuation token for the next page of results. This
	// will be absent if the end of the results list has been reached. If the token
	// is rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessProposals") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessProposals") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccessProposalsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccessProposalsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ModifyLabelsRequest: A request to modify the set of labels on a file. This
// request may contain many modifications that will either all succeed or all
// fail atomically.
type ModifyLabelsRequest struct {
	// Kind: This is always "drive#modifyLabelsRequest".
	Kind string `json:"kind,omitempty"`
	// LabelModifications: The list of modifications to apply to the labels on the
	// file.
	LabelModifications []*LabelModification `json:"labelModifications,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ModifyLabelsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ModifyLabelsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ModifyLabelsResponse: Response to a `ModifyLabels` request. This contains
// only those labels which were added or updated by the request.
type ModifyLabelsResponse struct {
	// Kind: This is always "drive#modifyLabelsResponse".
	Kind string `json:"kind,omitempty"`
	// ModifiedLabels: The list of labels which were added or updated by the
	// request.
	ModifiedLabels []*Label `json:"modifiedLabels,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ModifyLabelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ModifyLabelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Operation: This resource represents a long-running operation that is the
// result of a network API call.
type Operation struct {
	// Done: If the value is `false`, it means the operation is still in progress.
	// If `true`, the operation is completed, and either `error` or `response` is
	// available.
	Done bool `json:"done,omitempty"`
	// Error: The error result of the operation in case of failure or cancellation.
	Error *Status `json:"error,omitempty"`
	// Metadata: Service-specific metadata associated with the operation. It
	// typically contains progress information and common metadata such as create
	// time. Some services might not provide such metadata. Any method that returns
	// a long-running operation should document the metadata type, if any.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: The server-assigned name, which is only unique within the same service
	// that originally returns it. If you use the default HTTP mapping, the `name`
	// should be a resource name ending with `operations/{unique_id}`.
	Name string `json:"name,omitempty"`
	// Response: The normal, successful response of the operation. If the original
	// method returns no data on success, such as `Delete`, the response is
	// `google.protobuf.Empty`. If the original method is standard
	// `Get`/`Create`/`Update`, the response should be the resource. For other
	// methods, the response should have the type `XxxResponse`, where `Xxx` is the
	// original method name. For example, if the original method name is
	// `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
	Response googleapi.RawMessage `json:"response,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Done") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Done") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Operation) MarshalJSON() ([]byte, error) {
	type NoMethod Operation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Permission: A permission for a file. A permission grants a user, group,
// domain, or the world access to a file or a folder hierarchy. For more
// information, see Share files, folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
// By default, permission requests only return a subset of fields. Permission
// `kind`, `ID`, `type`, and `role` are always returned. To retrieve specific
// fields, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
// Some resource methods (such as `permissions.update`) require a
// `permissionId`. Use the `permissions.list` method to retrieve the ID for a
// file, folder, or shared drive.
type Permission struct {
	// AllowFileDiscovery: Whether the permission allows the file to be discovered
	// through search. This is only applicable for permissions of type `domain` or
	// `anyone`.
	AllowFileDiscovery bool `json:"allowFileDiscovery,omitempty"`
	// Deleted: Output only. Whether the account associated with this permission
	// has been deleted. This field only pertains to permissions of type `user` or
	// `group`.
	Deleted bool `json:"deleted,omitempty"`
	// DisplayName: Output only. The "pretty" name of the value of the permission.
	// The following is a list of examples for each type of permission: * `user` -
	// User's full name, as defined for their Google Account, such as "Dana A." *
	// `group` - Name of the Google Group, such as "The Company Administrators." *
	// `domain` - String domain name, such as "cymbalgroup.com." * `anyone` - No
	// `displayName` is present.
	DisplayName string `json:"displayName,omitempty"`
	// Domain: The domain to which this permission refers.
	Domain string `json:"domain,omitempty"`
	// EmailAddress: The email address of the user or group to which this
	// permission refers.
	EmailAddress string `json:"emailAddress,omitempty"`
	// ExpirationTime: The time at which this permission will expire (RFC 3339
	// date-time). Expiration times have the following restrictions: - They can
	// only be set on user and group permissions - The time must be in the future -
	// The time cannot be more than a year in the future
	ExpirationTime string `json:"expirationTime,omitempty"`
	// Id: Output only. The ID of this permission. This is a unique identifier for
	// the grantee, and is published in the User resource
	// (https://developers.google.com/workspace/drive/api/reference/rest/v3/User)
	// as `permissionId`. IDs should be treated as opaque values.
	Id string `json:"id,omitempty"`
	// InheritedPermissionsDisabled: When `true`, only organizers, owners, and
	// users with permissions added directly on the item can access it.
	InheritedPermissionsDisabled bool `json:"inheritedPermissionsDisabled,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#permission".
	Kind string `json:"kind,omitempty"`
	// PendingOwner: Whether the account associated with this permission is a
	// pending owner. Only populated for permissions of type `user` for files that
	// aren't in a shared drive.
	PendingOwner bool `json:"pendingOwner,omitempty"`
	// PermissionDetails: Output only. Details of whether the permissions on this
	// item are inherited or are directly on this item.
	PermissionDetails []*PermissionPermissionDetails `json:"permissionDetails,omitempty"`
	// PhotoLink: Output only. A link to the user's profile photo, if available.
	PhotoLink string `json:"photoLink,omitempty"`
	// Role: The role granted by this permission. Supported values include: *
	// `owner` * `organizer` * `fileOrganizer` * `writer` * `commenter` * `reader`
	// For more information, see Roles and permissions
	// (https://developers.google.com/workspace/drive/api/guides/ref-roles).
	Role string `json:"role,omitempty"`
	// TeamDrivePermissionDetails: Output only. Deprecated: Output only. Use
	// `permissionDetails` instead.
	TeamDrivePermissionDetails []*PermissionTeamDrivePermissionDetails `json:"teamDrivePermissionDetails,omitempty"`
	// Type: The type of the grantee. Supported values include: * `user` * `group`
	// * `domain` * `anyone` When creating a permission, if `type` is `user` or
	// `group`, you must provide an `emailAddress` for the user or group. If `type`
	// is `domain`, you must provide a `domain`. If `type` is `anyone`, no extra
	// information is required.
	Type string `json:"type,omitempty"`
	// View: Indicates the view for this permission. Only populated for permissions
	// that belong to a view. The only supported values are `published` and
	// `metadata`: * `published`: The permission's role is `publishedReader`. *
	// `metadata`: The item is only visible to the `metadata` view because the item
	// has limited access and the scope has at least read access to the parent. The
	// `metadata` view is only supported on folders. For more information, see
	// Views
	// (https://developers.google.com/workspace/drive/api/guides/ref-roles#views).
	View string `json:"view,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AllowFileDiscovery") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowFileDiscovery") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Permission) MarshalJSON() ([]byte, error) {
	type NoMethod Permission
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type PermissionPermissionDetails struct {
	// Inherited: Output only. Whether this permission is inherited. This field is
	// always populated. This is an output-only field.
	Inherited bool `json:"inherited,omitempty"`
	// InheritedFrom: Output only. The ID of the item from which this permission is
	// inherited. This is only populated for items in shared drives.
	InheritedFrom string `json:"inheritedFrom,omitempty"`
	// PermissionType: Output only. The permission type for this user. Supported
	// values include: * `file` * `member`
	PermissionType string `json:"permissionType,omitempty"`
	// Role: Output only. The primary role for this user. Supported values include:
	// * `owner` * `organizer` * `fileOrganizer` * `writer` * `commenter` *
	// `reader` For more information, see Roles and permissions
	// (https://developers.google.com/workspace/drive/api/guides/ref-roles).
	Role string `json:"role,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Inherited") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Inherited") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionPermissionDetails) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionPermissionDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type PermissionTeamDrivePermissionDetails struct {
	// Inherited: Deprecated: Output only. Use `permissionDetails/inherited`
	// instead.
	Inherited bool `json:"inherited,omitempty"`
	// InheritedFrom: Deprecated: Output only. Use
	// `permissionDetails/inheritedFrom` instead.
	InheritedFrom string `json:"inheritedFrom,omitempty"`
	// Role: Deprecated: Output only. Use `permissionDetails/role` instead.
	Role string `json:"role,omitempty"`
	// TeamDrivePermissionType: Deprecated: Output only. Use
	// `permissionDetails/permissionType` instead.
	TeamDrivePermissionType string `json:"teamDrivePermissionType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Inherited") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Inherited") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionTeamDrivePermissionDetails) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionTeamDrivePermissionDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PermissionList: A list of permissions for a file.
type PermissionList struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#permissionList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of permissions. This field
	// will be absent if the end of the permissions list has been reached. If the
	// token is rejected for any reason, it should be discarded, and pagination
	// should be restarted from the first page of results. The page token is
	// typically valid for several hours. However, if new items are added or
	// removed, your expected results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Permissions: The list of permissions. If `nextPageToken` is populated, then
	// this list may be incomplete and an additional page of results should be
	// fetched.
	Permissions []*Permission `json:"permissions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionList) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Reply: A reply to a comment on a file. Some resource methods (such as
// `replies.update`) require a `replyId`. Use the `replies.list` method to
// retrieve the ID for a reply.
type Reply struct {
	// Action: The action the reply performed to the parent comment. Valid values
	// are: * `resolve` * `reopen`
	Action string `json:"action,omitempty"`
	// Author: Output only. The author of the reply. The author's email address and
	// permission ID will not be populated.
	Author *User `json:"author,omitempty"`
	// Content: The plain text content of the reply. This field is used for setting
	// the content, while `htmlContent` should be displayed. This is required on
	// creates if no `action` is specified.
	Content string `json:"content,omitempty"`
	// CreatedTime: The time at which the reply was created (RFC 3339 date-time).
	CreatedTime string `json:"createdTime,omitempty"`
	// Deleted: Output only. Whether the reply has been deleted. A deleted reply
	// has no content.
	Deleted bool `json:"deleted,omitempty"`
	// HtmlContent: Output only. The content of the reply with HTML formatting.
	HtmlContent string `json:"htmlContent,omitempty"`
	// Id: Output only. The ID of the reply.
	Id string `json:"id,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#reply".
	Kind string `json:"kind,omitempty"`
	// ModifiedTime: The last time the reply was modified (RFC 3339 date-time).
	ModifiedTime string `json:"modifiedTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Reply) MarshalJSON() ([]byte, error) {
	type NoMethod Reply
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReplyList: A list of replies to a comment on a file.
type ReplyList struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#replyList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of replies. This will be
	// absent if the end of the replies list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results. The page token is typically valid
	// for several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Replies: The list of replies. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Replies []*Reply `json:"replies,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReplyList) MarshalJSON() ([]byte, error) {
	type NoMethod ReplyList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ResolveAccessProposalRequest: Request message for resolving an
// AccessProposal on a file.
type ResolveAccessProposalRequest struct {
	// Action: Required. The action to take on the access proposal.
	//
	// Possible values:
	//   "ACTION_UNSPECIFIED" - Unspecified action
	//   "ACCEPT" - The user accepts the access proposal. Note: If this action is
	// used, the `role` field must have at least one value.
	//   "DENY" - The user denies the access proposal.
	Action string `json:"action,omitempty"`
	// Role: Optional. The roles that the approver has allowed, if any. For more
	// information, see Roles and permissions
	// (https://developers.google.com/workspace/drive/api/guides/ref-roles). Note:
	// This field is required for the `ACCEPT` action.
	Role []string `json:"role,omitempty"`
	// SendNotification: Optional. Whether to send an email to the requester when
	// the access proposal is denied or accepted.
	SendNotification bool `json:"sendNotification,omitempty"`
	// View: Optional. Indicates the view for this access proposal. This should
	// only be set when the proposal belongs to a view. Only `published` is
	// supported.
	View string `json:"view,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ResolveAccessProposalRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ResolveAccessProposalRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Revision: The metadata for a revision to a file. Some resource methods (such
// as `revisions.update`) require a `revisionId`. Use the `revisions.list`
// method to retrieve the ID for a revision.
type Revision struct {
	// ExportLinks: Output only. Links for exporting Docs Editors files to specific
	// formats.
	ExportLinks map[string]string `json:"exportLinks,omitempty"`
	// Id: Output only. The ID of the revision.
	Id string `json:"id,omitempty"`
	// KeepForever: Whether to keep this revision forever, even if it is no longer
	// the head revision. If not set, the revision will be automatically purged 30
	// days after newer content is uploaded. This can be set on a maximum of 200
	// revisions for a file. This field is only applicable to files with binary
	// content in Drive.
	KeepForever bool `json:"keepForever,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string "drive#revision".
	Kind string `json:"kind,omitempty"`
	// LastModifyingUser: Output only. The last user to modify this revision. This
	// field is only populated when the last modification was performed by a
	// signed-in user.
	LastModifyingUser *User `json:"lastModifyingUser,omitempty"`
	// Md5Checksum: Output only. The MD5 checksum of the revision's content. This
	// is only applicable to files with binary content in Drive.
	Md5Checksum string `json:"md5Checksum,omitempty"`
	// MimeType: Output only. The MIME type of the revision.
	MimeType string `json:"mimeType,omitempty"`
	// ModifiedTime: The last time the revision was modified (RFC 3339 date-time).
	ModifiedTime string `json:"modifiedTime,omitempty"`
	// OriginalFilename: Output only. The original filename used to create this
	// revision. This is only applicable to files with binary content in Drive.
	OriginalFilename string `json:"originalFilename,omitempty"`
	// PublishAuto: Whether subsequent revisions will be automatically republished.
	// This is only applicable to Docs Editors files.
	PublishAuto bool `json:"publishAuto,omitempty"`
	// Published: Whether this revision is published. This is only applicable to
	// Docs Editors files.
	Published bool `json:"published,omitempty"`
	// PublishedLink: Output only. A link to the published revision. This is only
	// populated for Docs Editors files.
	PublishedLink string `json:"publishedLink,omitempty"`
	// PublishedOutsideDomain: Whether this revision is published outside the
	// domain. This is only applicable to Docs Editors files.
	PublishedOutsideDomain bool `json:"publishedOutsideDomain,omitempty"`
	// Size: Output only. The size of the revision's content in bytes. This is only
	// applicable to files with binary content in Drive.
	Size int64 `json:"size,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ExportLinks") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExportLinks") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Revision) MarshalJSON() ([]byte, error) {
	type NoMethod Revision
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RevisionList: A list of revisions of a file.
type RevisionList struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#revisionList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of revisions. This will be
	// absent if the end of the revisions list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results. The page token is typically valid
	// for several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Revisions: The list of revisions. If nextPageToken is populated, then this
	// list may be incomplete and an additional page of results should be fetched.
	Revisions []*Revision `json:"revisions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RevisionList) MarshalJSON() ([]byte, error) {
	type NoMethod RevisionList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type StartPageToken struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#startPageToken".
	Kind string `json:"kind,omitempty"`
	// StartPageToken: The starting page token for listing future changes. The page
	// token doesn't expire.
	StartPageToken string `json:"startPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StartPageToken) MarshalJSON() ([]byte, error) {
	type NoMethod StartPageToken
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each `Status` message contains three
// pieces of data: error code, error message, and error details. You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDrive: Deprecated: use the drive collection instead.
type TeamDrive struct {
	// BackgroundImageFile: An image file and cropping parameters from which a
	// background image for this Team Drive is set. This is a write only field; it
	// can only be set on `drive.teamdrives.update` requests that don't set
	// `themeId`. When specified, all fields of the `backgroundImageFile` must be
	// set.
	BackgroundImageFile *TeamDriveBackgroundImageFile `json:"backgroundImageFile,omitempty"`
	// BackgroundImageLink: A short-lived link to this Team Drive's background
	// image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// Capabilities: Capabilities the current user has on this Team Drive.
	Capabilities *TeamDriveCapabilities `json:"capabilities,omitempty"`
	// ColorRgb: The color of this Team Drive as an RGB hex string. It can only be
	// set on a `drive.teamdrives.update` request that does not set `themeId`.
	ColorRgb string `json:"colorRgb,omitempty"`
	// CreatedTime: The time at which the Team Drive was created (RFC 3339
	// date-time).
	CreatedTime string `json:"createdTime,omitempty"`
	// Id: The ID of this Team Drive which is also the ID of the top level folder
	// of this Team Drive.
	Id string `json:"id,omitempty"`
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#teamDrive".
	Kind string `json:"kind,omitempty"`
	// Name: The name of this Team Drive.
	Name string `json:"name,omitempty"`
	// OrgUnitId: The organizational unit of this shared drive. This field is only
	// populated on `drives.list` responses when the `useDomainAdminAccess`
	// parameter is set to `true`.
	OrgUnitId string `json:"orgUnitId,omitempty"`
	// Restrictions: A set of restrictions that apply to this Team Drive or items
	// inside this Team Drive.
	Restrictions *TeamDriveRestrictions `json:"restrictions,omitempty"`
	// ThemeId: The ID of the theme from which the background image and color will
	// be set. The set of possible `teamDriveThemes` can be retrieved from a
	// `drive.about.get` response. When not specified on a
	// `drive.teamdrives.create` request, a random theme is chosen from which the
	// background image and color are set. This is a write-only field; it can only
	// be set on requests that don't set `colorRgb` or `backgroundImageFile`.
	ThemeId string `json:"themeId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageFile") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageFile") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDrive) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDrive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveBackgroundImageFile: An image file and cropping parameters from
// which a background image for this Team Drive is set. This is a write only
// field; it can only be set on `drive.teamdrives.update` requests that don't
// set `themeId`. When specified, all fields of the `backgroundImageFile` must
// be set.
type TeamDriveBackgroundImageFile struct {
	// Id: The ID of an image file in Drive to use for the background image.
	Id string `json:"id,omitempty"`
	// Width: The width of the cropped image in the closed range of 0 to 1. This
	// value represents the width of the cropped image divided by the width of the
	// entire image. The height is computed by applying a width to height aspect
	// ratio of 80 to 9. The resulting image must be at least 1280 pixels wide and
	// 144 pixels high.
	Width float64 `json:"width,omitempty"`
	// XCoordinate: The X coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the horizontal distance from the left side of the entire
	// image to the left side of the cropping area divided by the width of the
	// entire image.
	XCoordinate float64 `json:"xCoordinate,omitempty"`
	// YCoordinate: The Y coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the vertical distance from the top side of the entire image
	// to the top side of the cropping area divided by the height of the entire
	// image.
	YCoordinate float64 `json:"yCoordinate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveBackgroundImageFile) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveBackgroundImageFile
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *TeamDriveBackgroundImageFile) UnmarshalJSON(data []byte) error {
	type NoMethod TeamDriveBackgroundImageFile
	var s1 struct {
		Width       gensupport.JSONFloat64 `json:"width"`
		XCoordinate gensupport.JSONFloat64 `json:"xCoordinate"`
		YCoordinate gensupport.JSONFloat64 `json:"yCoordinate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Width = float64(s1.Width)
	s.XCoordinate = float64(s1.XCoordinate)
	s.YCoordinate = float64(s1.YCoordinate)
	return nil
}

// TeamDriveCapabilities: Capabilities the current user has on this Team Drive.
type TeamDriveCapabilities struct {
	// CanAddChildren: Whether the current user can add children to folders in this
	// Team Drive.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanChangeCopyRequiresWriterPermissionRestriction: Whether the current user
	// can change the `copyRequiresWriterPermission` restriction of this Team
	// Drive.
	CanChangeCopyRequiresWriterPermissionRestriction bool `json:"canChangeCopyRequiresWriterPermissionRestriction,omitempty"`
	// CanChangeDomainUsersOnlyRestriction: Whether the current user can change the
	// `domainUsersOnly` restriction of this Team Drive.
	CanChangeDomainUsersOnlyRestriction bool `json:"canChangeDomainUsersOnlyRestriction,omitempty"`
	// CanChangeDownloadRestriction: Whether the current user can change
	// organizer-applied download restrictions of this shared drive.
	CanChangeDownloadRestriction bool `json:"canChangeDownloadRestriction,omitempty"`
	// CanChangeSharingFoldersRequiresOrganizerPermissionRestriction: Whether the
	// current user can change the `sharingFoldersRequiresOrganizerPermission`
	// restriction of this Team Drive.
	CanChangeSharingFoldersRequiresOrganizerPermissionRestriction bool `json:"canChangeSharingFoldersRequiresOrganizerPermissionRestriction,omitempty"`
	// CanChangeTeamDriveBackground: Whether the current user can change the
	// background of this Team Drive.
	CanChangeTeamDriveBackground bool `json:"canChangeTeamDriveBackground,omitempty"`
	// CanChangeTeamMembersOnlyRestriction: Whether the current user can change the
	// `teamMembersOnly` restriction of this Team Drive.
	CanChangeTeamMembersOnlyRestriction bool `json:"canChangeTeamMembersOnlyRestriction,omitempty"`
	// CanComment: Whether the current user can comment on files in this Team
	// Drive.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Whether the current user can copy files in this Team Drive.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDeleteChildren: Whether the current user can delete children from folders
	// in this Team Drive.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDeleteTeamDrive: Whether the current user can delete this Team Drive.
	// Attempting to delete the Team Drive may still fail if there are untrashed
	// items inside the Team Drive.
	CanDeleteTeamDrive bool `json:"canDeleteTeamDrive,omitempty"`
	// CanDownload: Whether the current user can download files in this Team Drive.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Whether the current user can edit files in this Team Drive
	CanEdit bool `json:"canEdit,omitempty"`
	// CanListChildren: Whether the current user can list the children of folders
	// in this Team Drive.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanManageMembers: Whether the current user can add members to this Team
	// Drive or remove them or change their role.
	CanManageMembers bool `json:"canManageMembers,omitempty"`
	// CanReadRevisions: Whether the current user can read the revisions resource
	// of files in this Team Drive.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanRemoveChildren: Deprecated: Use `canDeleteChildren` or `canTrashChildren`
	// instead.
	CanRemoveChildren bool `json:"canRemoveChildren,omitempty"`
	// CanRename: Whether the current user can rename files or folders in this Team
	// Drive.
	CanRename bool `json:"canRename,omitempty"`
	// CanRenameTeamDrive: Whether the current user can rename this Team Drive.
	CanRenameTeamDrive bool `json:"canRenameTeamDrive,omitempty"`
	// CanResetTeamDriveRestrictions: Whether the current user can reset the Team
	// Drive restrictions to defaults.
	CanResetTeamDriveRestrictions bool `json:"canResetTeamDriveRestrictions,omitempty"`
	// CanShare: Whether the current user can share files or folders in this Team
	// Drive.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrashChildren: Whether the current user can trash children from folders
	// in this Team Drive.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAddChildren") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAddChildren") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveRestrictions: A set of restrictions that apply to this Team Drive
// or items inside this Team Drive.
type TeamDriveRestrictions struct {
	// AdminManagedRestrictions: Whether administrative privileges on this Team
	// Drive are required to modify restrictions.
	AdminManagedRestrictions bool `json:"adminManagedRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download files inside this Team Drive, should be disabled for readers and
	// commenters. When this restriction is set to `true`, it will override the
	// similarly named field to `true` for any file inside this Team Drive.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// DomainUsersOnly: Whether access to this Team Drive and items inside this
	// Team Drive is restricted to users of the domain to which this Team Drive
	// belongs. This restriction may be overridden by other sharing policies
	// controlled outside of this Team Drive.
	DomainUsersOnly bool `json:"domainUsersOnly,omitempty"`
	// DownloadRestriction: Download restrictions applied by shared drive managers.
	DownloadRestriction *DownloadRestriction `json:"downloadRestriction,omitempty"`
	// SharingFoldersRequiresOrganizerPermission: If true, only users with the
	// organizer role can share folders. If false, users with either the organizer
	// role or the file organizer role can share folders.
	SharingFoldersRequiresOrganizerPermission bool `json:"sharingFoldersRequiresOrganizerPermission,omitempty"`
	// TeamMembersOnly: Whether access to items inside this Team Drive is
	// restricted to members of this Team Drive.
	TeamMembersOnly bool `json:"teamMembersOnly,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdminManagedRestrictions")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdminManagedRestrictions") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveRestrictions) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveRestrictions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveList: A list of Team Drives.
type TeamDriveList struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#teamDriveList".
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of Team Drives. This will be
	// absent if the end of the Team Drives list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results. The page token is typically valid
	// for several hours. However, if new items are added or removed, your expected
	// results might differ.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// TeamDrives: The list of Team Drives. If nextPageToken is populated, then
	// this list may be incomplete and an additional page of results should be
	// fetched.
	TeamDrives []*TeamDrive `json:"teamDrives,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveList) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// User: Information about a Drive user.
type User struct {
	// DisplayName: Output only. A plain text displayable name for this user.
	DisplayName string `json:"displayName,omitempty"`
	// EmailAddress: Output only. The email address of the user. This may not be
	// present in certain contexts if the user has not made their email address
	// visible to the requester.
	EmailAddress string `json:"emailAddress,omitempty"`
	// Kind: Output only. Identifies what kind of resource this is. Value: the
	// fixed string `drive#user`.
	Kind string `json:"kind,omitempty"`
	// Me: Output only. Whether this user is the requesting user.
	Me bool `json:"me,omitempty"`
	// PermissionId: Output only. The user's ID as visible in Permission resources.
	PermissionId string `json:"permissionId,omitempty"`
	// PhotoLink: Output only. A link to the user's profile photo, if available.
	PhotoLink string `json:"photoLink,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutGetCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the user, the user's Drive, and system
// capabilities. For more information, see Return user info
// (https://developers.google.com/workspace/drive/api/guides/user-info).
// Required: The `fields` parameter must be set. To return the exact fields you
// need, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
func (r *AboutService) Get() *AboutGetCall {
	c := &AboutGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AboutGetCall) Fields(s ...googleapi.Field) *AboutGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AboutGetCall) IfNoneMatch(entityTag string) *AboutGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AboutGetCall) Context(ctx context.Context) *AboutGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AboutGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AboutGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "about")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.about.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.about.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *About.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AboutGetCall) Do(opts ...googleapi.CallOption) (*About, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &About{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.about.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccessproposalsGetCall struct {
	s            *Service
	fileId       string
	proposalId   string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves an access proposal by ID. For more information, see Manage
// pending access proposals
// (https://developers.google.com/workspace/drive/api/guides/pending-access).
//
// - fileId: The ID of the item the request is on.
// - proposalId: The ID of the access proposal to resolve.
func (r *AccessproposalsService) Get(fileId string, proposalId string) *AccessproposalsGetCall {
	c := &AccessproposalsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.proposalId = proposalId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessproposalsGetCall) Fields(s ...googleapi.Field) *AccessproposalsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessproposalsGetCall) IfNoneMatch(entityTag string) *AccessproposalsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessproposalsGetCall) Context(ctx context.Context) *AccessproposalsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessproposalsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessproposalsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/accessproposals/{proposalId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"proposalId": c.proposalId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.accessproposals.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.accessproposals.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessProposal.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessproposalsGetCall) Do(opts ...googleapi.CallOption) (*AccessProposal, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessProposal{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.accessproposals.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccessproposalsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: List the access proposals on a file. For more information, see Manage
// pending access proposals
// (https://developers.google.com/workspace/drive/api/guides/pending-access).
// Note: Only approvers are able to list access proposals on a file. If the
// user isn't an approver, a 403 error is returned.
//
// - fileId: The ID of the item the request is on.
func (r *AccessproposalsService) List(fileId string) *AccessproposalsListCall {
	c := &AccessproposalsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// PageSize sets the optional parameter "pageSize": The number of results per
// page.
func (c *AccessproposalsListCall) PageSize(pageSize int64) *AccessproposalsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The continuation token on
// the list of access requests.
func (c *AccessproposalsListCall) PageToken(pageToken string) *AccessproposalsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessproposalsListCall) Fields(s ...googleapi.Field) *AccessproposalsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessproposalsListCall) IfNoneMatch(entityTag string) *AccessproposalsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessproposalsListCall) Context(ctx context.Context) *AccessproposalsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessproposalsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessproposalsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/accessproposals")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.accessproposals.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.accessproposals.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccessProposalsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessproposalsListCall) Do(opts ...googleapi.CallOption) (*ListAccessProposalsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccessProposalsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.accessproposals.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccessproposalsListCall) Pages(ctx context.Context, f func(*ListAccessProposalsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccessproposalsResolveCall struct {
	s                            *Service
	fileId                       string
	proposalId                   string
	resolveaccessproposalrequest *ResolveAccessProposalRequest
	urlParams_                   gensupport.URLParams
	ctx_                         context.Context
	header_                      http.Header
}

// Resolve: Approves or denies an access proposal. For more information, see
// Manage pending access proposals
// (https://developers.google.com/workspace/drive/api/guides/pending-access).
//
// - fileId: The ID of the item the request is on.
// - proposalId: The ID of the access proposal to resolve.
func (r *AccessproposalsService) Resolve(fileId string, proposalId string, resolveaccessproposalrequest *ResolveAccessProposalRequest) *AccessproposalsResolveCall {
	c := &AccessproposalsResolveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.proposalId = proposalId
	c.resolveaccessproposalrequest = resolveaccessproposalrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessproposalsResolveCall) Fields(s ...googleapi.Field) *AccessproposalsResolveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessproposalsResolveCall) Context(ctx context.Context) *AccessproposalsResolveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessproposalsResolveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessproposalsResolveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.resolveaccessproposalrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/accessproposals/{proposalId}:resolve")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"proposalId": c.proposalId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.accessproposals.resolve", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.accessproposals.resolve" call.
func (c *AccessproposalsResolveCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.accessproposals.resolve", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type AppsGetCall struct {
	s            *Service
	appId        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a specific app. For more information, see Return user info
// (https://developers.google.com/workspace/drive/api/guides/user-info).
//
// - appId: The ID of the app.
func (r *AppsService) Get(appId string) *AppsGetCall {
	c := &AppsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.appId = appId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AppsGetCall) Fields(s ...googleapi.Field) *AppsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AppsGetCall) IfNoneMatch(entityTag string) *AppsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AppsGetCall) Context(ctx context.Context) *AppsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AppsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AppsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "apps/{appId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"appId": c.appId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.apps.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.apps.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *App.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AppsGetCall) Do(opts ...googleapi.CallOption) (*App, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &App{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.apps.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AppsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a user's installed apps. For more information, see Return user
// info (https://developers.google.com/workspace/drive/api/guides/user-info).
func (r *AppsService) List() *AppsListCall {
	c := &AppsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// AppFilterExtensions sets the optional parameter "appFilterExtensions": A
// comma-separated list of file extensions to limit returned results. All
// results within the given app query scope which can open any of the given
// file extensions are included in the response. If `appFilterMimeTypes` are
// provided as well, the result is a union of the two resulting app lists.
func (c *AppsListCall) AppFilterExtensions(appFilterExtensions string) *AppsListCall {
	c.urlParams_.Set("appFilterExtensions", appFilterExtensions)
	return c
}

// AppFilterMimeTypes sets the optional parameter "appFilterMimeTypes": A
// comma-separated list of file extensions to limit returned results. All
// results within the given app query scope which can open any of the given
// MIME types will be included in the response. If `appFilterExtensions` are
// provided as well, the result is a union of the two resulting app lists.
func (c *AppsListCall) AppFilterMimeTypes(appFilterMimeTypes string) *AppsListCall {
	c.urlParams_.Set("appFilterMimeTypes", appFilterMimeTypes)
	return c
}

// LanguageCode sets the optional parameter "languageCode": A language or
// locale code, as defined by BCP 47, with some extensions from Unicode's LDML
// format (http://www.unicode.org/reports/tr35/).
func (c *AppsListCall) LanguageCode(languageCode string) *AppsListCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AppsListCall) Fields(s ...googleapi.Field) *AppsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AppsListCall) IfNoneMatch(entityTag string) *AppsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AppsListCall) Context(ctx context.Context) *AppsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AppsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AppsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "apps")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.apps.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.apps.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *AppList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AppsListCall) Do(opts ...googleapi.CallOption) (*AppList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AppList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.apps.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesGetStartPageTokenCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetStartPageToken: Gets the starting pageToken for listing future changes.
// For more information, see Retrieve changes
// (https://developers.google.com/workspace/drive/api/guides/manage-changes).
func (r *ChangesService) GetStartPageToken() *ChangesGetStartPageTokenCall {
	c := &ChangesGetStartPageTokenCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// DriveId sets the optional parameter "driveId": The ID of the shared drive
// for which the starting pageToken for listing future changes from that shared
// drive will be returned.
func (c *ChangesGetStartPageTokenCall) DriveId(driveId string) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesGetStartPageTokenCall) SupportsAllDrives(supportsAllDrives bool) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesGetStartPageTokenCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesGetStartPageTokenCall) TeamDriveId(teamDriveId string) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesGetStartPageTokenCall) Fields(s ...googleapi.Field) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChangesGetStartPageTokenCall) IfNoneMatch(entityTag string) *ChangesGetStartPageTokenCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesGetStartPageTokenCall) Context(ctx context.Context) *ChangesGetStartPageTokenCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesGetStartPageTokenCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesGetStartPageTokenCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes/startPageToken")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.getStartPageToken", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.getStartPageToken" call.
// Any non-2xx status code is an error. Response headers are in either
// *StartPageToken.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesGetStartPageTokenCall) Do(opts ...googleapi.CallOption) (*StartPageToken, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &StartPageToken{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.getStartPageToken", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the changes for a user or shared drive. For more information,
// see Retrieve changes
// (https://developers.google.com/workspace/drive/api/guides/manage-changes).
//
//   - pageToken: The token for continuing a previous list request on the next
//     page. This should be set to the value of 'nextPageToken' from the previous
//     response or to the response from the getStartPageToken method.
func (r *ChangesService) List(pageToken string) *ChangesListCall {
	c := &ChangesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// DriveId sets the optional parameter "driveId": The shared drive from which
// changes will be returned. If specified the change IDs will be reflective of
// the shared drive; use the combined drive ID and change ID as an identifier.
func (c *ChangesListCall) DriveId(driveId string) *ChangesListCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeCorpusRemovals sets the optional parameter "includeCorpusRemovals":
// Whether changes should include the file resource if the file is still
// accessible by the user at the time of the request, even when a file was
// removed from the list of changes and there will be no further change entries
// for this file.
func (c *ChangesListCall) IncludeCorpusRemovals(includeCorpusRemovals bool) *ChangesListCall {
	c.urlParams_.Set("includeCorpusRemovals", fmt.Sprint(includeCorpusRemovals))
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *ChangesListCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *ChangesListCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *ChangesListCall) IncludeLabels(includeLabels string) *ChangesListCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only 'published' is supported.
func (c *ChangesListCall) IncludePermissionsForView(includePermissionsForView string) *ChangesListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeRemoved sets the optional parameter "includeRemoved": Whether to
// include changes indicating that items have been removed from the list of
// changes, for example by deletion or loss of access.
func (c *ChangesListCall) IncludeRemoved(includeRemoved bool) *ChangesListCall {
	c.urlParams_.Set("includeRemoved", fmt.Sprint(includeRemoved))
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *ChangesListCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *ChangesListCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// changes to return per page.
func (c *ChangesListCall) PageSize(pageSize int64) *ChangesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// RestrictToMyDrive sets the optional parameter "restrictToMyDrive": Whether
// to restrict the results to changes inside the My Drive hierarchy. This omits
// changes to files such as those in the Application Data folder or shared
// files which have not been added to My Drive.
func (c *ChangesListCall) RestrictToMyDrive(restrictToMyDrive bool) *ChangesListCall {
	c.urlParams_.Set("restrictToMyDrive", fmt.Sprint(restrictToMyDrive))
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query within the corpora. Supported values are 'drive' and
// 'appDataFolder'.
func (c *ChangesListCall) Spaces(spaces string) *ChangesListCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesListCall) SupportsAllDrives(supportsAllDrives bool) *ChangesListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesListCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesListCall) TeamDriveId(teamDriveId string) *ChangesListCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesListCall) Fields(s ...googleapi.Field) *ChangesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChangesListCall) IfNoneMatch(entityTag string) *ChangesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesListCall) Context(ctx context.Context) *ChangesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ChangeList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesListCall) Do(opts ...googleapi.CallOption) (*ChangeList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ChangeList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesWatchCall struct {
	s          *Service
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Watch: Subscribes to changes for a user. For more information, see
// Notifications for resource changes
// (https://developers.google.com/workspace/drive/api/guides/push).
//
//   - pageToken: The token for continuing a previous list request on the next
//     page. This should be set to the value of 'nextPageToken' from the previous
//     response or to the response from the getStartPageToken method.
func (r *ChangesService) Watch(pageToken string, channel *Channel) *ChangesWatchCall {
	c := &ChangesWatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("pageToken", pageToken)
	c.channel = channel
	return c
}

// DriveId sets the optional parameter "driveId": The shared drive from which
// changes will be returned. If specified the change IDs will be reflective of
// the shared drive; use the combined drive ID and change ID as an identifier.
func (c *ChangesWatchCall) DriveId(driveId string) *ChangesWatchCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeCorpusRemovals sets the optional parameter "includeCorpusRemovals":
// Whether changes should include the file resource if the file is still
// accessible by the user at the time of the request, even when a file was
// removed from the list of changes and there will be no further change entries
// for this file.
func (c *ChangesWatchCall) IncludeCorpusRemovals(includeCorpusRemovals bool) *ChangesWatchCall {
	c.urlParams_.Set("includeCorpusRemovals", fmt.Sprint(includeCorpusRemovals))
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *ChangesWatchCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *ChangesWatchCall) IncludeLabels(includeLabels string) *ChangesWatchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only 'published' is supported.
func (c *ChangesWatchCall) IncludePermissionsForView(includePermissionsForView string) *ChangesWatchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeRemoved sets the optional parameter "includeRemoved": Whether to
// include changes indicating that items have been removed from the list of
// changes, for example by deletion or loss of access.
func (c *ChangesWatchCall) IncludeRemoved(includeRemoved bool) *ChangesWatchCall {
	c.urlParams_.Set("includeRemoved", fmt.Sprint(includeRemoved))
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *ChangesWatchCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *ChangesWatchCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// changes to return per page.
func (c *ChangesWatchCall) PageSize(pageSize int64) *ChangesWatchCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// RestrictToMyDrive sets the optional parameter "restrictToMyDrive": Whether
// to restrict the results to changes inside the My Drive hierarchy. This omits
// changes to files such as those in the Application Data folder or shared
// files which have not been added to My Drive.
func (c *ChangesWatchCall) RestrictToMyDrive(restrictToMyDrive bool) *ChangesWatchCall {
	c.urlParams_.Set("restrictToMyDrive", fmt.Sprint(restrictToMyDrive))
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query within the corpora. Supported values are 'drive' and
// 'appDataFolder'.
func (c *ChangesWatchCall) Spaces(spaces string) *ChangesWatchCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesWatchCall) SupportsAllDrives(supportsAllDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesWatchCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesWatchCall) TeamDriveId(teamDriveId string) *ChangesWatchCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesWatchCall) Fields(s ...googleapi.Field) *ChangesWatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesWatchCall) Context(ctx context.Context) *ChangesWatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesWatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesWatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes/watch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.watch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.watch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Channel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesWatchCall) Do(opts ...googleapi.CallOption) (*Channel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Channel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.watch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChannelsStopCall struct {
	s          *Service
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Stop: Stops watching resources through this channel. For more information,
// see Notifications for resource changes
// (https://developers.google.com/workspace/drive/api/guides/push).
func (r *ChannelsService) Stop(channel *Channel) *ChannelsStopCall {
	c := &ChannelsStopCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.channel = channel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChannelsStopCall) Fields(s ...googleapi.Field) *ChannelsStopCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChannelsStopCall) Context(ctx context.Context) *ChannelsStopCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChannelsStopCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChannelsStopCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "channels/stop")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.channels.stop", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.channels.stop" call.
func (c *ChannelsStopCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.channels.stop", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type CommentsCreateCall struct {
	s          *Service
	fileId     string
	comment    *Comment
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a comment on a file. For more information, see Manage
// comments and replies
// (https://developers.google.com/workspace/drive/api/guides/manage-comments).
// Required: The `fields` parameter must be set. To return the exact fields you
// need, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
//
// - fileId: The ID of the file.
func (r *CommentsService) Create(fileId string, comment *Comment) *CommentsCreateCall {
	c := &CommentsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.comment = comment
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsCreateCall) Fields(s ...googleapi.Field) *CommentsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsCreateCall) Context(ctx context.Context) *CommentsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.comment)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsCreateCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CommentsDeleteCall struct {
	s          *Service
	fileId     string
	commentId  string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a comment. For more information, see Manage comments and
// replies
// (https://developers.google.com/workspace/drive/api/guides/manage-comments).
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Delete(fileId string, commentId string) *CommentsDeleteCall {
	c := &CommentsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsDeleteCall) Fields(s ...googleapi.Field) *CommentsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsDeleteCall) Context(ctx context.Context) *CommentsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.delete" call.
func (c *CommentsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type CommentsGetCall struct {
	s            *Service
	fileId       string
	commentId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a comment by ID. For more information, see Manage comments and
// replies
// (https://developers.google.com/workspace/drive/api/guides/manage-comments).
// Required: The `fields` parameter must be set. To return the exact fields you
// need, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Get(fileId string, commentId string) *CommentsGetCall {
	c := &CommentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// return deleted comments. Deleted comments will not include their original
// content.
func (c *CommentsGetCall) IncludeDeleted(includeDeleted bool) *CommentsGetCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsGetCall) Fields(s ...googleapi.Field) *CommentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CommentsGetCall) IfNoneMatch(entityTag string) *CommentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsGetCall) Context(ctx context.Context) *CommentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsGetCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CommentsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's comments. For more information, see Manage comments and
// replies
// (https://developers.google.com/workspace/drive/api/guides/manage-comments).
// Required: The `fields` parameter must be set. To return the exact fields you
// need, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
//
// - fileId: The ID of the file.
func (r *CommentsService) List(fileId string) *CommentsListCall {
	c := &CommentsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// include deleted comments. Deleted comments will not include their original
// content.
func (c *CommentsListCall) IncludeDeleted(includeDeleted bool) *CommentsListCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// comments to return per page.
func (c *CommentsListCall) PageSize(pageSize int64) *CommentsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// 'nextPageToken' from the previous response.
func (c *CommentsListCall) PageToken(pageToken string) *CommentsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// StartModifiedTime sets the optional parameter "startModifiedTime": The
// minimum value of 'modifiedTime' for the result comments (RFC 3339
// date-time).
func (c *CommentsListCall) StartModifiedTime(startModifiedTime string) *CommentsListCall {
	c.urlParams_.Set("startModifiedTime", startModifiedTime)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsListCall) Fields(s ...googleapi.Field) *CommentsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CommentsListCall) IfNoneMatch(entityTag string) *CommentsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsListCall) Context(ctx context.Context) *CommentsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsListCall) Do(opts ...googleapi.CallOption) (*CommentList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *CommentsListCall) Pages(ctx context.Context, f func(*CommentList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type CommentsUpdateCall struct {
	s          *Service
	fileId     string
	commentId  string
	comment    *Comment
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a comment with patch semantics. For more information, see
// Manage comments and replies
// (https://developers.google.com/workspace/drive/api/guides/manage-comments).
// Required: The `fields` parameter must be set. To return the exact fields you
// need, see Return specific fields
// (https://developers.google.com/workspace/drive/api/guides/fields-parameter).
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Update(fileId string, commentId string, comment *Comment) *CommentsUpdateCall {
	c := &CommentsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.comment = comment
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsUpdateCall) Fields(s ...googleapi.Field) *CommentsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsUpdateCall) Context(ctx context.Context) *CommentsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.comment)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsUpdateCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesCreateCall struct {
	s          *Service
	drive      *Drive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a shared drive. For more information, see Manage shared
// drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
//   - requestId: An ID, such as a random UUID, which uniquely identifies this
//     user's request for idempotent creation of a shared drive. A repeated
//     request by the same user and with the same request ID will avoid creating
//     duplicates by attempting to create the same shared drive. If the shared
//     drive already exists a 409 error will be returned.
func (r *DrivesService) Create(requestId string, drive *Drive) *DrivesCreateCall {
	c := &DrivesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("requestId", requestId)
	c.drive = drive
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesCreateCall) Fields(s ...googleapi.Field) *DrivesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesCreateCall) Context(ctx context.Context) *DrivesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.drive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesCreateCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesDeleteCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a shared drive for which the user is an
// `organizer`. The shared drive cannot contain any untrashed items. For more
// information, see Manage shared drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Delete(driveId string) *DrivesDeleteCall {
	c := &DrivesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// AllowItemDeletion sets the optional parameter "allowItemDeletion": Whether
// any items inside the shared drive should also be deleted. This option is
// only supported when `useDomainAdminAccess` is also set to `true`.
func (c *DrivesDeleteCall) AllowItemDeletion(allowItemDeletion bool) *DrivesDeleteCall {
	c.urlParams_.Set("allowItemDeletion", fmt.Sprint(allowItemDeletion))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesDeleteCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesDeleteCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesDeleteCall) Fields(s ...googleapi.Field) *DrivesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesDeleteCall) Context(ctx context.Context) *DrivesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.delete" call.
func (c *DrivesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type DrivesGetCall struct {
	s            *Service
	driveId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a shared drive's metadata by ID. For more information, see Manage
// shared drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Get(driveId string) *DrivesGetCall {
	c := &DrivesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesGetCall) Fields(s ...googleapi.Field) *DrivesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DrivesGetCall) IfNoneMatch(entityTag string) *DrivesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesGetCall) Context(ctx context.Context) *DrivesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesGetCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesHideCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Hide: Hides a shared drive from the default view. For more information, see
// Manage shared drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Hide(driveId string) *DrivesHideCall {
	c := &DrivesHideCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesHideCall) Fields(s ...googleapi.Field) *DrivesHideCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesHideCall) Context(ctx context.Context) *DrivesHideCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesHideCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesHideCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}/hide")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.hide", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.hide" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesHideCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.hide", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the user's shared drives. This method accepts the `q`
// parameter, which is a search query combining one or more search terms. For
// more information, see the Search for shared drives
// (/workspace/drive/api/guides/search-shareddrives) guide.
func (r *DrivesService) List() *DrivesListCall {
	c := &DrivesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of shared
// drives to return per page.
func (c *DrivesListCall) PageSize(pageSize int64) *DrivesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for shared
// drives.
func (c *DrivesListCall) PageToken(pageToken string) *DrivesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": Query string for searching shared drives.
func (c *DrivesListCall) Q(q string) *DrivesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then all shared
// drives of the domain in which the requester is an administrator are
// returned.
func (c *DrivesListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesListCall) Fields(s ...googleapi.Field) *DrivesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DrivesListCall) IfNoneMatch(entityTag string) *DrivesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesListCall) Context(ctx context.Context) *DrivesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *DriveList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesListCall) Do(opts ...googleapi.CallOption) (*DriveList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DriveList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *DrivesListCall) Pages(ctx context.Context, f func(*DriveList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type DrivesUnhideCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Unhide: Restores a shared drive to the default view. For more information,
// see Manage shared drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Unhide(driveId string) *DrivesUnhideCall {
	c := &DrivesUnhideCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesUnhideCall) Fields(s ...googleapi.Field) *DrivesUnhideCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesUnhideCall) Context(ctx context.Context) *DrivesUnhideCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesUnhideCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesUnhideCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}/unhide")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.unhide", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.unhide" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesUnhideCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.unhide", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesUpdateCall struct {
	s          *Service
	driveId    string
	drive      *Drive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates the metadata for a shared drive. For more information, see
// Manage shared drives
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives).
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Update(driveId string, drive *Drive) *DrivesUpdateCall {
	c := &DrivesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	c.drive = drive
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesUpdateCall) Fields(s ...googleapi.Field) *DrivesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesUpdateCall) Context(ctx context.Context) *DrivesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.drive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesUpdateCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesCopyCall struct {
	s          *Service
	fileId     string
	file       *File
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Copy: Creates a copy of a file and applies any requested updates with patch
// semantics. For more information, see Create and manage files
// (https://developers.google.com/workspace/drive/api/guides/create-file).
//
// - fileId: The ID of the file.
func (r *FilesService) Copy(fileId string, file *File) *FilesCopyCall {
	c := &FilesCopyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.file = file
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Copying files into multiple folders is no longer supported. Use
// shortcuts instead.
func (c *FilesCopyCall) EnforceSingleParent(enforceSingleParent bool) *FilesCopyCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IgnoreDefaultVisibility sets the optional parameter
// "ignoreDefaultVisibility": Whether to ignore the domain's default visibility
// settings for the created file. Domain administrators can choose to make all
// uploaded files visible to the domain by default; this parameter bypasses
// that behavior for the request. Permissions are still inherited from parent
// folders.
func (c *FilesCopyCall) IgnoreDefaultVisibility(ignoreDefaultVisibility bool) *FilesCopyCall {
	c.urlParams_.Set("ignoreDefaultVisibility", fmt.Sprint(ignoreDefaultVisibility))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesCopyCall) IncludeLabels(includeLabels string) *FilesCopyCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesCopyCall) IncludePermissionsForView(includePermissionsForView string) *FilesCopyCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// KeepRevisionForever sets the optional parameter "keepRevisionForever":
// Whether to set the `keepForever` field in the new head revision. This is
// only applicable to files with binary content in Google Drive. Only 200
// revisions for the file can be kept forever. If the limit is reached, try
// deleting pinned revisions.
func (c *FilesCopyCall) KeepRevisionForever(keepRevisionForever bool) *FilesCopyCall {
	c.urlParams_.Set("keepRevisionForever", fmt.Sprint(keepRevisionForever))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": A language hint for
// OCR processing during image import (ISO 639-1 code).
func (c *FilesCopyCall) OcrLanguage(ocrLanguage string) *FilesCopyCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesCopyCall) SupportsAllDrives(supportsAllDrives bool) *FilesCopyCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesCopyCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesCopyCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesCopyCall) Fields(s ...googleapi.Field) *FilesCopyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesCopyCall) Context(ctx context.Context) *FilesCopyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesCopyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesCopyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/copy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.copy", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.copy" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesCopyCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.copy", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesCreateCall struct {
	s          *Service
	file       *File
	urlParams_ gensupport.URLParams
	mediaInfo_ *gensupport.MediaInfo
	ctx_       context.Context
	header_    http.Header
}

// Create:  Creates a file. For more information, see Create and manage files
// (/workspace/drive/api/guides/create-file). This method supports an */upload*
// URI and accepts uploaded media with the following characteristics: -
// *Maximum file size:* 5,120 GB - *Accepted Media MIME types:* `*/*` (Specify
// a valid MIME type, rather than the literal `*/*` value. The literal `*/*` is
// only used to indicate that any valid MIME type can be uploaded. For more
// information, see Google Workspace and Google Drive supported MIME types
// (/workspace/drive/api/guides/mime-types).) For more information on uploading
// files, see Upload file data (/workspace/drive/api/guides/manage-uploads).
// Apps creating shortcuts with the `create` method must specify the MIME type
// `application/vnd.google-apps.shortcut`. Apps should specify a file extension
// in the `name` property when inserting files with the API. For example, an
// operation to insert a JPEG file should specify something like "name":
// "cat.jpg" in the metadata. Subsequent `GET` requests include the read-only
// `fileExtension` property populated with the extension originally specified
// in the `name` property. When a Google Drive user requests to download a
// file, or when the file is downloaded through the sync client, Drive builds a
// full filename (with extension) based on the name. In cases where the
// extension is missing, Drive attempts to determine the extension based on the
// file's MIME type.
func (r *FilesService) Create(file *File) *FilesCreateCall {
	c := &FilesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.file = file
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Creating files in multiple folders is no longer supported.
func (c *FilesCreateCall) EnforceSingleParent(enforceSingleParent bool) *FilesCreateCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IgnoreDefaultVisibility sets the optional parameter
// "ignoreDefaultVisibility": Whether to ignore the domain's default visibility
// settings for the created file. Domain administrators can choose to make all
// uploaded files visible to the domain by default; this parameter bypasses
// that behavior for the request. Permissions are still inherited from parent
// folders.
func (c *FilesCreateCall) IgnoreDefaultVisibility(ignoreDefaultVisibility bool) *FilesCreateCall {
	c.urlParams_.Set("ignoreDefaultVisibility", fmt.Sprint(ignoreDefaultVisibility))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesCreateCall) IncludeLabels(includeLabels string) *FilesCreateCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesCreateCall) IncludePermissionsForView(includePermissionsForView string) *FilesCreateCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// KeepRevisionForever sets the optional parameter "keepRevisionForever":
// Whether to set the `keepForever` field in the new head revision. This is
// only applicable to files with binary content in Google Drive. Only 200
// revisions for the file can be kept forever. If the limit is reached, try
// deleting pinned revisions.
func (c *FilesCreateCall) KeepRevisionForever(keepRevisionForever bool) *FilesCreateCall {
	c.urlParams_.Set("keepRevisionForever", fmt.Sprint(keepRevisionForever))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": A language hint for
// OCR processing during image import (ISO 639-1 code).
func (c *FilesCreateCall) OcrLanguage(ocrLanguage string) *FilesCreateCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesCreateCall) SupportsAllDrives(supportsAllDrives bool) *FilesCreateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesCreateCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesCreateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseContentAsIndexableText sets the optional parameter
// "useContentAsIndexableText": Whether to use the uploaded content as
// indexable text.
func (c *FilesCreateCall) UseContentAsIndexableText(useContentAsIndexableText bool) *FilesCreateCall {
	c.urlParams_.Set("useContentAsIndexableText", fmt.Sprint(useContentAsIndexableText))
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *FilesCreateCall) Media(r io.Reader, options ...googleapi.MediaOption) *FilesCreateCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *FilesCreateCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *FilesCreateCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *FilesCreateCall) ProgressUpdater(pu googleapi.ProgressUpdater) *FilesCreateCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesCreateCall) Fields(s ...googleapi.Field) *FilesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *FilesCreateCall) Context(ctx context.Context) *FilesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/drive/v3/files")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesCreateCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesDeleteCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a file owned by the user without moving it to
// the trash. For more information, see Trash or delete files and folders
// (https://developers.google.com/workspace/drive/api/guides/delete). If the
// file belongs to a shared drive, the user must be an `organizer` on the
// parent folder. If the target is a folder, all descendants owned by the user
// are also deleted.
//
// - fileId: The ID of the file.
func (r *FilesService) Delete(fileId string) *FilesDeleteCall {
	c := &FilesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item isn't in a shared drive and its last parent is
// deleted but the item itself isn't, the item will be placed under its owner's
// root.
func (c *FilesDeleteCall) EnforceSingleParent(enforceSingleParent bool) *FilesDeleteCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesDeleteCall) SupportsAllDrives(supportsAllDrives bool) *FilesDeleteCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesDeleteCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesDeleteCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesDeleteCall) Fields(s ...googleapi.Field) *FilesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesDeleteCall) Context(ctx context.Context) *FilesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.delete" call.
func (c *FilesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesDownloadCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Download: Downloads the content of a file. For more information, see
// Download and export files
// (https://developers.google.com/workspace/drive/api/guides/manage-downloads).
// Operations are valid for 24 hours from the time of creation.
//
// - fileId: The ID of the file to download.
func (r *FilesService) Download(fileId string) *FilesDownloadCall {
	c := &FilesDownloadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// MimeType sets the optional parameter "mimeType": The MIME type the file
// should be downloaded as. This field can only be set when downloading Google
// Workspace documents. For a list of supported MIME types, see Export MIME
// types for Google Workspace documents
// (/workspace/drive/api/guides/ref-export-formats). If not set, a Google
// Workspace document is downloaded with a default MIME type. The default MIME
// type might change in the future.
func (c *FilesDownloadCall) MimeType(mimeType string) *FilesDownloadCall {
	c.urlParams_.Set("mimeType", mimeType)
	return c
}

// RevisionId sets the optional parameter "revisionId": The revision ID of the
// file to download. This field can only be set when downloading blob files,
// Google Docs, and Google Sheets. Returns `INVALID_ARGUMENT` if downloading a
// specific revision on the file is unsupported.
func (c *FilesDownloadCall) RevisionId(revisionId string) *FilesDownloadCall {
	c.urlParams_.Set("revisionId", revisionId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesDownloadCall) Fields(s ...googleapi.Field) *FilesDownloadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesDownloadCall) Context(ctx context.Context) *FilesDownloadCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesDownloadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesDownloadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/download")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.download", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.download" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesDownloadCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.download", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesEmptyTrashCall struct {
	s          *Service
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// EmptyTrash: Permanently deletes all of the user's trashed files. For more
// information, see Trash or delete files and folders
// (https://developers.google.com/workspace/drive/api/guides/delete).
func (r *FilesService) EmptyTrash() *FilesEmptyTrashCall {
	c := &FilesEmptyTrashCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// DriveId sets the optional parameter "driveId": If set, empties the trash of
// the provided shared drive.
func (c *FilesEmptyTrashCall) DriveId(driveId string) *FilesEmptyTrashCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item isn't in a shared drive and its last parent is
// deleted but the item itself isn't, the item will be placed under its owner's
// root.
func (c *FilesEmptyTrashCall) EnforceSingleParent(enforceSingleParent bool) *FilesEmptyTrashCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesEmptyTrashCall) Fields(s ...googleapi.Field) *FilesEmptyTrashCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesEmptyTrashCall) Context(ctx context.Context) *FilesEmptyTrashCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesEmptyTrashCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesEmptyTrashCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/trash")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.emptyTrash", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.emptyTrash" call.
func (c *FilesEmptyTrashCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.emptyTrash", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesExportCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Export: Exports a Google Workspace document to the requested MIME type and
// returns exported byte content. For more information, see Download and export
// files
// (https://developers.google.com/workspace/drive/api/guides/manage-downloads).
// Note that the exported content is limited to 10 MB.
//
//   - fileId: The ID of the file.
//   - mimeType: The MIME type of the format requested for this export. For a
//     list of supported MIME types, see Export MIME types for Google Workspace
//     documents (/workspace/drive/api/guides/ref-export-formats).
func (r *FilesService) Export(fileId string, mimeType string) *FilesExportCall {
	c := &FilesExportCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.urlParams_.Set("mimeType", mimeType)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesExportCall) Fields(s ...googleapi.Field) *FilesExportCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesExportCall) IfNoneMatch(entityTag string) *FilesExportCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *FilesExportCall) Context(ctx context.Context) *FilesExportCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesExportCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesExportCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/export")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.export", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *FilesExportCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "drive.files.export" call.
func (c *FilesExportCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.export", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesGenerateIdsCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GenerateIds: Generates a set of file IDs which can be provided in create or
// copy requests. For more information, see Create and manage files
// (https://developers.google.com/workspace/drive/api/guides/create-file).
func (r *FilesService) GenerateIds() *FilesGenerateIdsCall {
	c := &FilesGenerateIdsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Count sets the optional parameter "count": The number of IDs to return.
func (c *FilesGenerateIdsCall) Count(count int64) *FilesGenerateIdsCall {
	c.urlParams_.Set("count", fmt.Sprint(count))
	return c
}

// Space sets the optional parameter "space": The space in which the IDs can be
// used to create files. Supported values are `drive` and `appDataFolder`.
// (Default: `drive`.) For more information, see File organization
// (https://developers.google.com/workspace/drive/api/guides/about-files#file-organization).
func (c *FilesGenerateIdsCall) Space(space string) *FilesGenerateIdsCall {
	c.urlParams_.Set("space", space)
	return c
}

// Type sets the optional parameter "type": The type of items which the IDs can
// be used for. Supported values are `files` and `shortcuts`. Note that
// `shortcuts` are only supported in the `drive` `space`. (Default: `files`.)
// For more information, see File organization
// (https://developers.google.com/workspace/drive/api/guides/about-files#file-organization).
func (c *FilesGenerateIdsCall) Type(type_ string) *FilesGenerateIdsCall {
	c.urlParams_.Set("type", type_)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesGenerateIdsCall) Fields(s ...googleapi.Field) *FilesGenerateIdsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesGenerateIdsCall) IfNoneMatch(entityTag string) *FilesGenerateIdsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesGenerateIdsCall) Context(ctx context.Context) *FilesGenerateIdsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesGenerateIdsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesGenerateIdsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/generateIds")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.generateIds", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.generateIds" call.
// Any non-2xx status code is an error. Response headers are in either
// *GeneratedIds.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesGenerateIdsCall) Do(opts ...googleapi.CallOption) (*GeneratedIds, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GeneratedIds{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.generateIds", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesGetCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get:  Gets a file's metadata or content by ID. For more information, see
// Search for files and folders (/workspace/drive/api/guides/search-files). If
// you provide the URL parameter `alt=media`, then the response includes the
// file contents in the response body. Downloading content with `alt=media`
// only works if the file is stored in Drive. To download Google Docs, Sheets,
// and Slides use `files.export`
// (/workspace/drive/api/reference/rest/v3/files/export) instead. For more
// information, see Download and export files
// (/workspace/drive/api/guides/manage-downloads).
//
// - fileId: The ID of the file.
func (r *FilesService) Get(fileId string) *FilesGetCall {
	c := &FilesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// AcknowledgeAbuse sets the optional parameter "acknowledgeAbuse": Whether the
// user is acknowledging the risk of downloading known malware or other abusive
// files. This is only applicable when the `alt` parameter is set to `media`
// and the user is the owner of the file or an organizer of the shared drive in
// which the file resides.
func (c *FilesGetCall) AcknowledgeAbuse(acknowledgeAbuse bool) *FilesGetCall {
	c.urlParams_.Set("acknowledgeAbuse", fmt.Sprint(acknowledgeAbuse))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesGetCall) IncludeLabels(includeLabels string) *FilesGetCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesGetCall) IncludePermissionsForView(includePermissionsForView string) *FilesGetCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesGetCall) SupportsAllDrives(supportsAllDrives bool) *FilesGetCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesGetCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesGetCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesGetCall) Fields(s ...googleapi.Field) *FilesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesGetCall) IfNoneMatch(entityTag string) *FilesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *FilesGetCall) Context(ctx context.Context) *FilesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *FilesGetCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "drive.files.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesGetCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the user's files. For more information, see Search for files
// and folders (/workspace/drive/api/guides/search-files). This method accepts
// the `q` parameter, which is a search query combining one or more search
// terms. This method returns *all* files by default, including trashed files.
// If you don't want trashed files to appear in the list, use the
// `trashed=false` query parameter to remove trashed files from the results.
func (r *FilesService) List() *FilesListCall {
	c := &FilesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Corpora sets the optional parameter "corpora": Bodies of items (files or
// documents) to which the query applies. Supported bodies are: * `user` *
// `domain` * `drive` * `allDrives` Prefer `user` or `drive` to `allDrives` for
// efficiency. By default, corpora is set to `user`. However, this can change
// depending on the filter set through the `q` parameter. For more information,
// see File organization
// (https://developers.google.com/workspace/drive/api/guides/about-files#file-organization).
func (c *FilesListCall) Corpora(corpora string) *FilesListCall {
	c.urlParams_.Set("corpora", corpora)
	return c
}

// Corpus sets the optional parameter "corpus": Deprecated: The source of files
// to list. Use `corpora` instead.
//
// Possible values:
//
//	"domain" - Files shared to the user's domain.
//	"user" - Files owned by or shared to the user.
func (c *FilesListCall) Corpus(corpus string) *FilesListCall {
	c.urlParams_.Set("corpus", corpus)
	return c
}

// DriveId sets the optional parameter "driveId": ID of the shared drive to
// search.
func (c *FilesListCall) DriveId(driveId string) *FilesListCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *FilesListCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *FilesListCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesListCall) IncludeLabels(includeLabels string) *FilesListCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesListCall) IncludePermissionsForView(includePermissionsForView string) *FilesListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *FilesListCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *FilesListCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// OrderBy sets the optional parameter "orderBy": A comma-separated list of
// sort keys. Valid keys are: * `createdTime`: When the file was created. *
// `folder`: The folder ID. This field is sorted using alphabetical ordering. *
// `modifiedByMeTime`: The last time the file was modified by the user. *
// `modifiedTime`: The last time the file was modified by anyone. * `name`: The
// name of the file. This field is sorted using alphabetical ordering, so 1,
// 12, 2, 22. * `name_natural`: The name of the file. This field is sorted
// using natural sort ordering, so 1, 2, 12, 22. * `quotaBytesUsed`: The number
// of storage quota bytes used by the file. * `recency`: The most recent
// timestamp from the file's date-time fields. * `sharedWithMeTime`: When the
// file was shared with the user, if applicable. * `starred`: Whether the user
// has starred the file. * `viewedByMeTime`: The last time the file was viewed
// by the user. Each key sorts ascending by default, but can be reversed with
// the `desc` modifier. Example usage: `?orderBy=folder,modifiedTime
// desc,name`.
func (c *FilesListCall) OrderBy(orderBy string) *FilesListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of files
// to return per page. Partial or empty result pages are possible even before
// the end of the files list has been reached.
func (c *FilesListCall) PageSize(pageSize int64) *FilesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response.
func (c *FilesListCall) PageToken(pageToken string) *FilesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": A query for filtering the file results.
// For supported syntax, see Search for files and folders
// (/workspace/drive/api/guides/search-files).
func (c *FilesListCall) Q(q string) *FilesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query within the corpora. Supported values are `drive` and
// `appDataFolder`. For more information, see File organization
// (https://developers.google.com/workspace/drive/api/guides/about-files#file-organization).
func (c *FilesListCall) Spaces(spaces string) *FilesListCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesListCall) SupportsAllDrives(supportsAllDrives bool) *FilesListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesListCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *FilesListCall) TeamDriveId(teamDriveId string) *FilesListCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesListCall) Fields(s ...googleapi.Field) *FilesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesListCall) IfNoneMatch(entityTag string) *FilesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesListCall) Context(ctx context.Context) *FilesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *FileList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesListCall) Do(opts ...googleapi.CallOption) (*FileList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &FileList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *FilesListCall) Pages(ctx context.Context, f func(*FileList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type FilesListLabelsCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListLabels: Lists the labels on a file. For more information, see List
// labels on a file
// (https://developers.google.com/workspace/drive/api/guides/list-labels).
//
// - fileId: The ID for the file.
func (r *FilesService) ListLabels(fileId string) *FilesListLabelsCall {
	c := &FilesListLabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// labels to return per page. When not set, defaults to 100.
func (c *FilesListLabelsCall) MaxResults(maxResults int64) *FilesListLabelsCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response.
func (c *FilesListLabelsCall) PageToken(pageToken string) *FilesListLabelsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesListLabelsCall) Fields(s ...googleapi.Field) *FilesListLabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesListLabelsCall) IfNoneMatch(entityTag string) *FilesListLabelsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesListLabelsCall) Context(ctx context.Context) *FilesListLabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesListLabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesListLabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/listLabels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.listLabels", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.listLabels" call.
// Any non-2xx status code is an error. Response headers are in either
// *LabelList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesListLabelsCall) Do(opts ...googleapi.CallOption) (*LabelList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LabelList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.listLabels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *FilesListLabelsCall) Pages(ctx context.Context, f func(*LabelList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type FilesModifyLabelsCall struct {
	s                   *Service
	fileId              string
	modifylabelsrequest *ModifyLabelsRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// ModifyLabels: Modifies the set of labels applied to a file. For more
// information, see Set a label field on a file
// (https://developers.google.com/workspace/drive/api/guides/set-label).
// Returns a list of the labels that were added or modified.
//
// - fileId: The ID of the file to which the labels belong.
func (r *FilesService) ModifyLabels(fileId string, modifylabelsrequest *ModifyLabelsRequest) *FilesModifyLabelsCall {
	c := &FilesModifyLabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.modifylabelsrequest = modifylabelsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesModifyLabelsCall) Fields(s ...googleapi.Field) *FilesModifyLabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesModifyLabelsCall) Context(ctx context.Context) *FilesModifyLabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesModifyLabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesModifyLabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.modifylabelsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/modifyLabels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.modifyLabels", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.modifyLabels" call.
// Any non-2xx status code is an error. Response headers are in either
// *ModifyLabelsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FilesModifyLabelsCall) Do(opts ...googleapi.CallOption) (*ModifyLabelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ModifyLabelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.modifyLabels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesUpdateCall struct {
	s          *Service
	fileId     string
	file       *File
	urlParams_ gensupport.URLParams
	mediaInfo_ *gensupport.MediaInfo
	ctx_       context.Context
	header_    http.Header
}

// Update:  Updates a file's metadata, content, or both. When calling this
// method, only populate fields in the request that you want to modify. When
// updating fields, some fields might be changed automatically, such as
// `modifiedDate`. This method supports patch semantics. This method supports
// an */upload* URI and accepts uploaded media with the following
// characteristics: - *Maximum file size:* 5,120 GB - *Accepted Media MIME
// types:* `*/*` (Specify a valid MIME type, rather than the literal `*/*`
// value. The literal `*/*` is only used to indicate that any valid MIME type
// can be uploaded. For more information, see Google Workspace and Google Drive
// supported MIME types (/workspace/drive/api/guides/mime-types).) For more
// information on uploading files, see Upload file data
// (/workspace/drive/api/guides/manage-uploads).
//
// - fileId: The ID of the file.
func (r *FilesService) Update(fileId string, file *File) *FilesUpdateCall {
	c := &FilesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.file = file
	return c
}

// AddParents sets the optional parameter "addParents": A comma-separated list
// of parent IDs to add.
func (c *FilesUpdateCall) AddParents(addParents string) *FilesUpdateCall {
	c.urlParams_.Set("addParents", addParents)
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Adding files to multiple folders is no longer supported. Use
// shortcuts instead.
func (c *FilesUpdateCall) EnforceSingleParent(enforceSingleParent bool) *FilesUpdateCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesUpdateCall) IncludeLabels(includeLabels string) *FilesUpdateCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesUpdateCall) IncludePermissionsForView(includePermissionsForView string) *FilesUpdateCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// KeepRevisionForever sets the optional parameter "keepRevisionForever":
// Whether to set the `keepForever` field in the new head revision. This is
// only applicable to files with binary content in Google Drive. Only 200
// revisions for the file can be kept forever. If the limit is reached, try
// deleting pinned revisions.
func (c *FilesUpdateCall) KeepRevisionForever(keepRevisionForever bool) *FilesUpdateCall {
	c.urlParams_.Set("keepRevisionForever", fmt.Sprint(keepRevisionForever))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": A language hint for
// OCR processing during image import (ISO 639-1 code).
func (c *FilesUpdateCall) OcrLanguage(ocrLanguage string) *FilesUpdateCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// RemoveParents sets the optional parameter "removeParents": A comma-separated
// list of parent IDs to remove.
func (c *FilesUpdateCall) RemoveParents(removeParents string) *FilesUpdateCall {
	c.urlParams_.Set("removeParents", removeParents)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesUpdateCall) SupportsAllDrives(supportsAllDrives bool) *FilesUpdateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesUpdateCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesUpdateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseContentAsIndexableText sets the optional parameter
// "useContentAsIndexableText": Whether to use the uploaded content as
// indexable text.
func (c *FilesUpdateCall) UseContentAsIndexableText(useContentAsIndexableText bool) *FilesUpdateCall {
	c.urlParams_.Set("useContentAsIndexableText", fmt.Sprint(useContentAsIndexableText))
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *FilesUpdateCall) Media(r io.Reader, options ...googleapi.MediaOption) *FilesUpdateCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *FilesUpdateCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *FilesUpdateCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *FilesUpdateCall) ProgressUpdater(pu googleapi.ProgressUpdater) *FilesUpdateCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesUpdateCall) Fields(s ...googleapi.Field) *FilesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *FilesUpdateCall) Context(ctx context.Context) *FilesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/drive/v3/files/{fileId}")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesUpdateCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesWatchCall struct {
	s          *Service
	fileId     string
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Watch: Subscribes to changes to a file. For more information, see
// Notifications for resource changes
// (https://developers.google.com/workspace/drive/api/guides/push).
//
// - fileId: The ID of the file.
func (r *FilesService) Watch(fileId string, channel *Channel) *FilesWatchCall {
	c := &FilesWatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.channel = channel
	return c
}

// AcknowledgeAbuse sets the optional parameter "acknowledgeAbuse": Whether the
// user is acknowledging the risk of downloading known malware or other abusive
// files. This is only applicable when the `alt` parameter is set to `media`
// and the user is the owner of the file or an organizer of the shared drive in
// which the file resides.
func (c *FilesWatchCall) AcknowledgeAbuse(acknowledgeAbuse bool) *FilesWatchCall {
	c.urlParams_.Set("acknowledgeAbuse", fmt.Sprint(acknowledgeAbuse))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesWatchCall) IncludeLabels(includeLabels string) *FilesWatchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesWatchCall) IncludePermissionsForView(includePermissionsForView string) *FilesWatchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesWatchCall) SupportsAllDrives(supportsAllDrives bool) *FilesWatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesWatchCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesWatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesWatchCall) Fields(s ...googleapi.Field) *FilesWatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesWatchCall) Context(ctx context.Context) *FilesWatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesWatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesWatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/watch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.watch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.watch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Channel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesWatchCall) Do(opts ...googleapi.CallOption) (*Channel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Channel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.watch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OperationsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest state of a long-running operation. Clients can use this
// method to poll the operation result at intervals as recommended by the API
// service.
//
// - name: The name of the operation resource.
func (r *OperationsService) Get(name string) *OperationsGetCall {
	c := &OperationsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsGetCall) Fields(s ...googleapi.Field) *OperationsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OperationsGetCall) IfNoneMatch(entityTag string) *OperationsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsGetCall) Context(ctx context.Context) *OperationsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "operations/{name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.operations.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.operations.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsGetCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.operations.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsCreateCall struct {
	s          *Service
	fileId     string
	permission *Permission
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a permission for a file or shared drive. For more
// information, see Share files, folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
// **Warning:** Concurrent permissions operations on the same file aren't
// supported; only the last update is applied.
//
// - fileId: The ID of the file or shared drive.
func (r *PermissionsService) Create(fileId string, permission *Permission) *PermissionsCreateCall {
	c := &PermissionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permission = permission
	return c
}

// EmailMessage sets the optional parameter "emailMessage": A plain text custom
// message to include in the notification email.
func (c *PermissionsCreateCall) EmailMessage(emailMessage string) *PermissionsCreateCall {
	c.urlParams_.Set("emailMessage", emailMessage)
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsCreateCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsCreateCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: See `moveToNewOwnersRoot` for details.
func (c *PermissionsCreateCall) EnforceSingleParent(enforceSingleParent bool) *PermissionsCreateCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// MoveToNewOwnersRoot sets the optional parameter "moveToNewOwnersRoot": This
// parameter only takes effect if the item isn't in a shared drive and the
// request is attempting to transfer the ownership of the item. If set to
// `true`, the item is moved to the new owner's My Drive root folder and all
// prior parents removed. If set to `false`, parents aren't changed.
func (c *PermissionsCreateCall) MoveToNewOwnersRoot(moveToNewOwnersRoot bool) *PermissionsCreateCall {
	c.urlParams_.Set("moveToNewOwnersRoot", fmt.Sprint(moveToNewOwnersRoot))
	return c
}

// SendNotificationEmail sets the optional parameter "sendNotificationEmail":
// Whether to send a notification email when sharing to users or groups. This
// defaults to `true` for users and groups, and is not allowed for other
// requests. It must not be disabled for ownership transfers.
func (c *PermissionsCreateCall) SendNotificationEmail(sendNotificationEmail bool) *PermissionsCreateCall {
	c.urlParams_.Set("sendNotificationEmail", fmt.Sprint(sendNotificationEmail))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsCreateCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsCreateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsCreateCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsCreateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TransferOwnership sets the optional parameter "transferOwnership": Whether
// to transfer ownership to the specified user and downgrade the current owner
// to a writer. This parameter is required as an acknowledgement of the side
// effect. For more information, see Transfer file ownership
// (https://developers.google.com/workspace/drive/api/guides/transfer-file).
func (c *PermissionsCreateCall) TransferOwnership(transferOwnership bool) *PermissionsCreateCall {
	c.urlParams_.Set("transferOwnership", fmt.Sprint(transferOwnership))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator. If set to `true`, and if the
// following additional conditions are met, the requester is granted access: 1.
// The file ID parameter refers to a shared drive. 2. The requester is an
// administrator of the domain to which the shared drive belongs. For more
// information, see Manage shared drives as domain administrators
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives#manage-administrators).
func (c *PermissionsCreateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsCreateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsCreateCall) Fields(s ...googleapi.Field) *PermissionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsCreateCall) Context(ctx context.Context) *PermissionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.permission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsCreateCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsDeleteCall struct {
	s            *Service
	fileId       string
	permissionId string
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Delete: Deletes a permission. For more information, see Share files,
// folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
// **Warning:** Concurrent permissions operations on the same file aren't
// supported; only the last update is applied.
//
// - fileId: The ID of the file or shared drive.
// - permissionId: The ID of the permission.
func (r *PermissionsService) Delete(fileId string, permissionId string) *PermissionsDeleteCall {
	c := &PermissionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsDeleteCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsDeleteCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsDeleteCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsDeleteCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsDeleteCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsDeleteCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator. If set to `true`, and if the
// following additional conditions are met, the requester is granted access: 1.
// The file ID parameter refers to a shared drive. 2. The requester is an
// administrator of the domain to which the shared drive belongs. For more
// information, see Manage shared drives as domain administrators
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives#manage-administrators).
func (c *PermissionsDeleteCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsDeleteCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsDeleteCall) Fields(s ...googleapi.Field) *PermissionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsDeleteCall) Context(ctx context.Context) *PermissionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.delete" call.
func (c *PermissionsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type PermissionsGetCall struct {
	s            *Service
	fileId       string
	permissionId string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a permission by ID. For more information, see Share files,
// folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
//
// - fileId: The ID of the file.
// - permissionId: The ID of the permission.
func (r *PermissionsService) Get(fileId string, permissionId string) *PermissionsGetCall {
	c := &PermissionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsGetCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsGetCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsGetCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsGetCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator. If set to `true`, and if the
// following additional conditions are met, the requester is granted access: 1.
// The file ID parameter refers to a shared drive. 2. The requester is an
// administrator of the domain to which the shared drive belongs. For more
// information, see Manage shared drives as domain administrators
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives#manage-administrators).
func (c *PermissionsGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsGetCall) Fields(s ...googleapi.Field) *PermissionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PermissionsGetCall) IfNoneMatch(entityTag string) *PermissionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsGetCall) Context(ctx context.Context) *PermissionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsGetCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's or shared drive's permissions. For more information,
// see Share files, folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
//
// - fileId: The ID of the file or shared drive.
func (r *PermissionsService) List(fileId string) *PermissionsListCall {
	c := &PermissionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *PermissionsListCall) IncludePermissionsForView(includePermissionsForView string) *PermissionsListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// permissions to return per page. When not set for files in a shared drive, at
// most 100 results will be returned. When not set for files that are not in a
// shared drive, the entire list will be returned.
func (c *PermissionsListCall) PageSize(pageSize int64) *PermissionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response.
func (c *PermissionsListCall) PageToken(pageToken string) *PermissionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsListCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsListCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator. If set to `true`, and if the
// following additional conditions are met, the requester is granted access: 1.
// The file ID parameter refers to a shared drive. 2. The requester is an
// administrator of the domain to which the shared drive belongs. For more
// information, see Manage shared drives as domain administrators
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives#manage-administrators).
func (c *PermissionsListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsListCall) Fields(s ...googleapi.Field) *PermissionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PermissionsListCall) IfNoneMatch(entityTag string) *PermissionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsListCall) Context(ctx context.Context) *PermissionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *PermissionList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsListCall) Do(opts ...googleapi.CallOption) (*PermissionList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PermissionList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *PermissionsListCall) Pages(ctx context.Context, f func(*PermissionList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type PermissionsUpdateCall struct {
	s            *Service
	fileId       string
	permissionId string
	permission   *Permission
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Update: Updates a permission with patch semantics. For more information, see
// Share files, folders, and drives
// (https://developers.google.com/workspace/drive/api/guides/manage-sharing).
// **Warning:** Concurrent permissions operations on the same file aren't
// supported; only the last update is applied.
//
// - fileId: The ID of the file or shared drive.
// - permissionId: The ID of the permission.
func (r *PermissionsService) Update(fileId string, permissionId string, permission *Permission) *PermissionsUpdateCall {
	c := &PermissionsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	c.permission = permission
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsUpdateCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsUpdateCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// RemoveExpiration sets the optional parameter "removeExpiration": Whether to
// remove the expiration date.
func (c *PermissionsUpdateCall) RemoveExpiration(removeExpiration bool) *PermissionsUpdateCall {
	c.urlParams_.Set("removeExpiration", fmt.Sprint(removeExpiration))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsUpdateCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsUpdateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsUpdateCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsUpdateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TransferOwnership sets the optional parameter "transferOwnership": Whether
// to transfer ownership to the specified user and downgrade the current owner
// to a writer. This parameter is required as an acknowledgement of the side
// effect. For more information, see Transfer file ownership
// (https://developers.google.com//workspace/drive/api/guides/transfer-file).
func (c *PermissionsUpdateCall) TransferOwnership(transferOwnership bool) *PermissionsUpdateCall {
	c.urlParams_.Set("transferOwnership", fmt.Sprint(transferOwnership))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator. If set to `true`, and if the
// following additional conditions are met, the requester is granted access: 1.
// The file ID parameter refers to a shared drive. 2. The requester is an
// administrator of the domain to which the shared drive belongs. For more
// information, see Manage shared drives as domain administrators
// (https://developers.google.com/workspace/drive/api/guides/manage-shareddrives#manage-administrators).
func (c *PermissionsUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsUpdateCall) Fields(s ...googleapi.Field) *PermissionsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsUpdateCall) Context(ctx context.Context) *PermissionsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.permission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsUpdateCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesCreateCall struct {
	s          *Service
	fileId     string
	commentId  string
	reply      *Reply
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a reply to a comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *RepliesService) Create(fileId string, commentId string, reply *Reply) *RepliesCreateCall {
	c := &RepliesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.reply = reply
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesCreateCall) Fields(s ...googleapi.Field) *RepliesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesCreateCall) Context(ctx context.Context) *RepliesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.reply)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Reply.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesCreateCall) Do(opts ...googleapi.CallOption) (*Reply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Reply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesDeleteCall struct {
	s          *Service
	fileId     string
	commentId  string
	replyId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a reply.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Delete(fileId string, commentId string, replyId string) *RepliesDeleteCall {
	c := &RepliesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesDeleteCall) Fields(s ...googleapi.Field) *RepliesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesDeleteCall) Context(ctx context.Context) *RepliesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.delete" call.
func (c *RepliesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type RepliesGetCall struct {
	s            *Service
	fileId       string
	commentId    string
	replyId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a reply by ID.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Get(fileId string, commentId string, replyId string) *RepliesGetCall {
	c := &RepliesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// return deleted replies. Deleted replies will not include their original
// content.
func (c *RepliesGetCall) IncludeDeleted(includeDeleted bool) *RepliesGetCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesGetCall) Fields(s ...googleapi.Field) *RepliesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RepliesGetCall) IfNoneMatch(entityTag string) *RepliesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesGetCall) Context(ctx context.Context) *RepliesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Reply.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesGetCall) Do(opts ...googleapi.CallOption) (*Reply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Reply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesListCall struct {
	s            *Service
	fileId       string
	commentId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a comment's replies.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *RepliesService) List(fileId string, commentId string) *RepliesListCall {
	c := &RepliesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// include deleted replies. Deleted replies will not include their original
// content.
func (c *RepliesListCall) IncludeDeleted(includeDeleted bool) *RepliesListCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// replies to return per page.
func (c *RepliesListCall) PageSize(pageSize int64) *RepliesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// 'nextPageToken' from the previous response.
func (c *RepliesListCall) PageToken(pageToken string) *RepliesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesListCall) Fields(s ...googleapi.Field) *RepliesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RepliesListCall) IfNoneMatch(entityTag string) *RepliesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesListCall) Context(ctx context.Context) *RepliesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ReplyList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesListCall) Do(opts ...googleapi.CallOption) (*ReplyList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ReplyList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *RepliesListCall) Pages(ctx context.Context, f func(*ReplyList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type RepliesUpdateCall struct {
	s          *Service
	fileId     string
	commentId  string
	replyId    string
	reply      *Reply
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a reply with patch semantics.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Update(fileId string, commentId string, replyId string, reply *Reply) *RepliesUpdateCall {
	c := &RepliesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	c.reply = reply
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesUpdateCall) Fields(s ...googleapi.Field) *RepliesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesUpdateCall) Context(ctx context.Context) *RepliesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.reply)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Reply.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesUpdateCall) Do(opts ...googleapi.CallOption) (*Reply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Reply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RevisionsDeleteCall struct {
	s          *Service
	fileId     string
	revisionId string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a file version. You can only delete revisions
// for files with binary content in Google Drive, like images or videos.
// Revisions for other files, like Google Docs or Sheets, and the last
// remaining file version can't be deleted. For more information, see Manage
// file revisions
// (https://developers.google.com/drive/api/guides/manage-revisions).
//
// - fileId: The ID of the file.
// - revisionId: The ID of the revision.
func (r *RevisionsService) Delete(fileId string, revisionId string) *RevisionsDeleteCall {
	c := &RevisionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsDeleteCall) Fields(s ...googleapi.Field) *RevisionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsDeleteCall) Context(ctx context.Context) *RevisionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.delete" call.
func (c *RevisionsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type RevisionsGetCall struct {
	s            *Service
	fileId       string
	revisionId   string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a revision's metadata or content by ID. For more information, see
// Manage file revisions
// (https://developers.google.com/workspace/drive/api/guides/manage-revisions).
//
// - fileId: The ID of the file.
// - revisionId: The ID of the revision.
func (r *RevisionsService) Get(fileId string, revisionId string) *RevisionsGetCall {
	c := &RevisionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	return c
}

// AcknowledgeAbuse sets the optional parameter "acknowledgeAbuse": Whether the
// user is acknowledging the risk of downloading known malware or other abusive
// files. This is only applicable when the `alt` parameter is set to `media`
// and the user is the owner of the file or an organizer of the shared drive in
// which the file resides.
func (c *RevisionsGetCall) AcknowledgeAbuse(acknowledgeAbuse bool) *RevisionsGetCall {
	c.urlParams_.Set("acknowledgeAbuse", fmt.Sprint(acknowledgeAbuse))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsGetCall) Fields(s ...googleapi.Field) *RevisionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RevisionsGetCall) IfNoneMatch(entityTag string) *RevisionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *RevisionsGetCall) Context(ctx context.Context) *RevisionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *RevisionsGetCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "drive.revisions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Revision.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsGetCall) Do(opts ...googleapi.CallOption) (*Revision, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Revision{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RevisionsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's revisions. For more information, see Manage file
// revisions
// (https://developers.google.com/workspace/drive/api/guides/manage-revisions).
//
// - fileId: The ID of the file.
func (r *RevisionsService) List(fileId string) *RevisionsListCall {
	c := &RevisionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// revisions to return per page.
func (c *RevisionsListCall) PageSize(pageSize int64) *RevisionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// 'nextPageToken' from the previous response.
func (c *RevisionsListCall) PageToken(pageToken string) *RevisionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsListCall) Fields(s ...googleapi.Field) *RevisionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RevisionsListCall) IfNoneMatch(entityTag string) *RevisionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsListCall) Context(ctx context.Context) *RevisionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *RevisionList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsListCall) Do(opts ...googleapi.CallOption) (*RevisionList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RevisionList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *RevisionsListCall) Pages(ctx context.Context, f func(*RevisionList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type RevisionsUpdateCall struct {
	s          *Service
	fileId     string
	revisionId string
	revision   *Revision
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a revision with patch semantics. For more information, see
// Manage file revisions
// (https://developers.google.com/workspace/drive/api/guides/manage-revisions).
//
// - fileId: The ID of the file.
// - revisionId: The ID of the revision.
func (r *RevisionsService) Update(fileId string, revisionId string, revision *Revision) *RevisionsUpdateCall {
	c := &RevisionsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	c.revision = revision
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsUpdateCall) Fields(s ...googleapi.Field) *RevisionsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsUpdateCall) Context(ctx context.Context) *RevisionsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.revision)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Revision.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsUpdateCall) Do(opts ...googleapi.CallOption) (*Revision, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Revision{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesCreateCall struct {
	s          *Service
	teamdrive  *TeamDrive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Deprecated: Use `drives.create` instead.
//
//   - requestId: An ID, such as a random UUID, which uniquely identifies this
//     user's request for idempotent creation of a Team Drive. A repeated request
//     by the same user and with the same request ID will avoid creating
//     duplicates by attempting to create the same Team Drive. If the Team Drive
//     already exists a 409 error will be returned.
func (r *TeamdrivesService) Create(requestId string, teamdrive *TeamDrive) *TeamdrivesCreateCall {
	c := &TeamdrivesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("requestId", requestId)
	c.teamdrive = teamdrive
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesCreateCall) Fields(s ...googleapi.Field) *TeamdrivesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesCreateCall) Context(ctx context.Context) *TeamdrivesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.teamdrive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesCreateCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesDeleteCall struct {
	s           *Service
	teamDriveId string
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Delete: Deprecated: Use `drives.delete` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Delete(teamDriveId string) *TeamdrivesDeleteCall {
	c := &TeamdrivesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesDeleteCall) Fields(s ...googleapi.Field) *TeamdrivesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesDeleteCall) Context(ctx context.Context) *TeamdrivesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.delete" call.
func (c *TeamdrivesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type TeamdrivesGetCall struct {
	s            *Service
	teamDriveId  string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Deprecated: Use `drives.get` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Get(teamDriveId string) *TeamdrivesGetCall {
	c := &TeamdrivesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the Team Drive belongs.
func (c *TeamdrivesGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesGetCall) Fields(s ...googleapi.Field) *TeamdrivesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TeamdrivesGetCall) IfNoneMatch(entityTag string) *TeamdrivesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesGetCall) Context(ctx context.Context) *TeamdrivesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesGetCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Deprecated: Use `drives.list` instead.
func (r *TeamdrivesService) List() *TeamdrivesListCall {
	c := &TeamdrivesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of Team
// Drives to return.
func (c *TeamdrivesListCall) PageSize(pageSize int64) *TeamdrivesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for Team
// Drives.
func (c *TeamdrivesListCall) PageToken(pageToken string) *TeamdrivesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": Query string for searching Team Drives.
func (c *TeamdrivesListCall) Q(q string) *TeamdrivesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then all Team
// Drives of the domain in which the requester is an administrator are
// returned.
func (c *TeamdrivesListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesListCall) Fields(s ...googleapi.Field) *TeamdrivesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TeamdrivesListCall) IfNoneMatch(entityTag string) *TeamdrivesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesListCall) Context(ctx context.Context) *TeamdrivesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDriveList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesListCall) Do(opts ...googleapi.CallOption) (*TeamDriveList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDriveList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *TeamdrivesListCall) Pages(ctx context.Context, f func(*TeamDriveList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type TeamdrivesUpdateCall struct {
	s           *Service
	teamDriveId string
	teamdrive   *TeamDrive
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Update: Deprecated: Use `drives.update` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Update(teamDriveId string, teamdrive *TeamDrive) *TeamdrivesUpdateCall {
	c := &TeamdrivesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	c.teamdrive = teamdrive
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the Team Drive belongs.
func (c *TeamdrivesUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesUpdateCall) Fields(s ...googleapi.Field) *TeamdrivesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesUpdateCall) Context(ctx context.Context) *TeamdrivesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.teamdrive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesUpdateCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
