# K6

Since testcontainers-go <a href="https://github.com/testcontainers/testcontainers-go/releases/tag/v0.26.0"><span class="tc-version">:material-tag: v0.26.0</span></a>

## Introduction

The Testcontainers module for K6.

### Using k6 extensions

This module takes advantage of [k6x](https://github.com/szkiba/k6x) to dynamically build a `k6` binary with all the [k6 extensions](https://k6.io/docs/extensions/get-started/explore/) required by the test script.

## Adding this module to your project dependencies

Please run the following command to add the K6 module to your Go dependencies:

```
go get github.com/testcontainers/testcontainers-go/modules/k6
```

## Usage example

<!--codeinclude-->
[Creating a httpbin application](../../modules/k6/examples_test.go) inside_block:runHTTPBin
[Obtain IP for the httpbin application](../../modules/k6/examples_test.go) inside_block:getHTTPBinIP
[k6 script for testing httpbin](../../modules/k6/scripts/httpbin.js)
[Creating a K6 container](../../modules/k6/examples_test.go) inside_block:runK6Container
<!--/codeinclude-->

## Module Reference

### Run function

- Since testcontainers-go <a href="https://github.com/testcontainers/testcontainers-go/releases/tag/v0.32.0"><span class="tc-version">:material-tag: v0.32.0</span></a>

!!!info
    The `RunContainer(ctx, opts...)` function is deprecated and will be removed in the next major release of _Testcontainers for Go_.

The K6 module exposes one entrypoint function to run the K6 container, and this function receives three parameters:

```golang
func Run(ctx context.Context, img string, opts ...testcontainers.ContainerCustomizer) (*K6Container, error)
```

- `context.Context`, the Go context.
- `string`, the Docker image to use.
- `testcontainers.ContainerCustomizer`, a variadic argument for passing options.

### Container Options

When starting the K6 container, you can pass options in a variadic way to configure it.

#### Image

!!! warning
    The K6 module uses a `k6x` image to build a `k6` binary with all the required extensions. Therefore, only the [szkiba/k6x](https://hub.docker.com/r/szkiba/k6x) image should be used with this module.

Use the second argument in the `Run` function to set a valid Docker image.
In example: `Run(context.Background(), "szkiba/k6x:v0.3.1")`.

{% include "../features/common_functional_options.md" %}

#### SetEnvVar

`SetEnvVar` sets an [environment variable](https://k6.io/docs/using-k6/environment-variables/) for the test script using the '--env' command-line flag in the k6 command in the container.

```golang
k6.New(ctx, k6.SetEnvVar("URL","test.k6.io"), k6.WithTestScript("/tests/test.js"))
```

#### WithCache

Use `WithCache` sets a volume to be used as [cache for building the k6 binary](https://github.com/szkiba/k6x#cache) inside the `k6` container.
This option improves considerably the execution time of test suites that creates multiple `k6` test containers.

By default, a new volume is created and automatically removed when the test session ends.

This is convenient for example for CI/CD environments. In other cases, such as local testing, it can be convenient to reuse the same cache volume across test sessions. In this case, the TC_K6_BUILD_CACHE environment variables can be used to provide the name of a volume to be used and kept across test sessions. If this volume does not exist, it will be created.

```golang
k6.New(ctx, WithCache(), k6.WithTestScript("/tests/test.js"))
```

#### WithCmdOptions

Use `WithCmdOptions` to pass a variadic list of strings as [options](https://k6.io/docs/using-k6/k6-options/reference/) to the k6 run command

```golang
k6.New(ctx, WithCmdOptions("--vus=10", "--duration=30s"), k6.WithTestScript("/tests/test.js"))
```

#### WithTestScript

Use the `WithTestScript` option to specify the test script to run. The path to the script must be an absolute path. This option copies the script file to the container and pass it to k6's `run` command. At least one `WithTestScript` or `WithRemoteTestScript`  option must be specified.

```golang
k6.New(ctx, k6.WithTestScript("/tests/test.js"))
```

#### WithRemoteTestScript

Use the `WithRemoteTestScript` option to specify the remote test script to run. The path to the remote script must be a http or https url. Basic authentication is supported. This option performs a HTTP `GET` to copy the remote file locally then copies the script file to the container and pass it to k6's `run` command. The default timeout for the `GET` is 60 seconds. Only javascript, or more specifically `Content-Type:text/javascript` is supported.  At least one `WithTestScript` or `WithRemoteTestScript`  option must be specified.

```golang
scriptUrl:="https://raw.githubusercontent.com/testcontainers/testcontainers-go/main/modules/k6/scripts/pass.js"

uri, _ := url.Parse(scriptUrl)
desc := k6.DownloadableFile{Uri: *uri , DownloadDir: t.TempDir()}
options := k6.WithRemoteTestScript(desc)

k6.New(ctx, k6.WithCache(), options)
```

### Container Methods

The K6 container does not expose any method.
